﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Filter.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 */
class DescribeDBClusterEndpointsRequest : public RDSRequest {
 public:
  AWS_RDS_API DescribeDBClusterEndpointsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeDBClusterEndpoints"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The DB cluster identifier of the DB cluster associated with the endpoint.
   * This parameter is stored as a lowercase string.</p>
   */
  inline const Aws::String& GetDBClusterIdentifier() const { return m_dBClusterIdentifier; }
  inline bool DBClusterIdentifierHasBeenSet() const { return m_dBClusterIdentifierHasBeenSet; }
  template <typename DBClusterIdentifierT = Aws::String>
  void SetDBClusterIdentifier(DBClusterIdentifierT&& value) {
    m_dBClusterIdentifierHasBeenSet = true;
    m_dBClusterIdentifier = std::forward<DBClusterIdentifierT>(value);
  }
  template <typename DBClusterIdentifierT = Aws::String>
  DescribeDBClusterEndpointsRequest& WithDBClusterIdentifier(DBClusterIdentifierT&& value) {
    SetDBClusterIdentifier(std::forward<DBClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the endpoint to describe. This parameter is stored as a
   * lowercase string.</p>
   */
  inline const Aws::String& GetDBClusterEndpointIdentifier() const { return m_dBClusterEndpointIdentifier; }
  inline bool DBClusterEndpointIdentifierHasBeenSet() const { return m_dBClusterEndpointIdentifierHasBeenSet; }
  template <typename DBClusterEndpointIdentifierT = Aws::String>
  void SetDBClusterEndpointIdentifier(DBClusterEndpointIdentifierT&& value) {
    m_dBClusterEndpointIdentifierHasBeenSet = true;
    m_dBClusterEndpointIdentifier = std::forward<DBClusterEndpointIdentifierT>(value);
  }
  template <typename DBClusterEndpointIdentifierT = Aws::String>
  DescribeDBClusterEndpointsRequest& WithDBClusterEndpointIdentifier(DBClusterEndpointIdentifierT&& value) {
    SetDBClusterEndpointIdentifier(std::forward<DBClusterEndpointIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A set of name-value pairs that define which endpoints to include in the
   * output. The filters are specified as name-value pairs, in the format
   * <code>Name=<i>endpoint_type</i>,Values=<i>endpoint_type1</i>,<i>endpoint_type2</i>,...</code>.
   * <code>Name</code> can be one of: <code>db-cluster-endpoint-type</code>,
   * <code>db-cluster-endpoint-custom-type</code>,
   * <code>db-cluster-endpoint-id</code>, <code>db-cluster-endpoint-status</code>.
   * <code>Values</code> for the <code> db-cluster-endpoint-type</code> filter can be
   * one or more of: <code>reader</code>, <code>writer</code>, <code>custom</code>.
   * <code>Values</code> for the <code>db-cluster-endpoint-custom-type</code> filter
   * can be one or more of: <code>reader</code>, <code>any</code>.
   * <code>Values</code> for the <code>db-cluster-endpoint-status</code> filter can
   * be one or more of: <code>available</code>, <code>creating</code>,
   * <code>deleting</code>, <code>inactive</code>, <code>modifying</code>.</p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeDBClusterEndpointsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeDBClusterEndpointsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a pagination token
   * called a marker is included in the response so you can retrieve the remaining
   * results.</p> <p>Default: 100</p> <p>Constraints: Minimum 20, maximum 100.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeDBClusterEndpointsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous
   * <code>DescribeDBClusterEndpoints</code> request. If this parameter is specified,
   * the response includes only records beyond the marker, up to the value specified
   * by <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeDBClusterEndpointsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBClusterIdentifier;

  Aws::String m_dBClusterEndpointIdentifier;

  Aws::Vector<Filter> m_filters;

  int m_maxRecords{0};

  Aws::String m_marker;
  bool m_dBClusterIdentifierHasBeenSet = false;
  bool m_dBClusterEndpointIdentifierHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
