﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/Document.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/qbusiness/QBusinessRequest.h>
#include <aws/qbusiness/QBusiness_EXPORTS.h>
#include <aws/qbusiness/model/DataSourceVpcConfiguration.h>
#include <aws/qbusiness/model/DocumentEnrichmentConfiguration.h>
#include <aws/qbusiness/model/MediaExtractionConfiguration.h>

#include <utility>

namespace Aws {
namespace QBusiness {
namespace Model {

/**
 */
class UpdateDataSourceRequest : public QBusinessRequest {
 public:
  AWS_QBUSINESS_API UpdateDataSourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateDataSource"; }

  AWS_QBUSINESS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> The identifier of the Amazon Q Business application the data source is
   * attached to.</p>
   */
  inline const Aws::String& GetApplicationId() const { return m_applicationId; }
  inline bool ApplicationIdHasBeenSet() const { return m_applicationIdHasBeenSet; }
  template <typename ApplicationIdT = Aws::String>
  void SetApplicationId(ApplicationIdT&& value) {
    m_applicationIdHasBeenSet = true;
    m_applicationId = std::forward<ApplicationIdT>(value);
  }
  template <typename ApplicationIdT = Aws::String>
  UpdateDataSourceRequest& WithApplicationId(ApplicationIdT&& value) {
    SetApplicationId(std::forward<ApplicationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the index attached to the data source connector.</p>
   */
  inline const Aws::String& GetIndexId() const { return m_indexId; }
  inline bool IndexIdHasBeenSet() const { return m_indexIdHasBeenSet; }
  template <typename IndexIdT = Aws::String>
  void SetIndexId(IndexIdT&& value) {
    m_indexIdHasBeenSet = true;
    m_indexId = std::forward<IndexIdT>(value);
  }
  template <typename IndexIdT = Aws::String>
  UpdateDataSourceRequest& WithIndexId(IndexIdT&& value) {
    SetIndexId(std::forward<IndexIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the data source connector.</p>
   */
  inline const Aws::String& GetDataSourceId() const { return m_dataSourceId; }
  inline bool DataSourceIdHasBeenSet() const { return m_dataSourceIdHasBeenSet; }
  template <typename DataSourceIdT = Aws::String>
  void SetDataSourceId(DataSourceIdT&& value) {
    m_dataSourceIdHasBeenSet = true;
    m_dataSourceId = std::forward<DataSourceIdT>(value);
  }
  template <typename DataSourceIdT = Aws::String>
  UpdateDataSourceRequest& WithDataSourceId(DataSourceIdT&& value) {
    SetDataSourceId(std::forward<DataSourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A name of the data source connector.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  UpdateDataSourceRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline Aws::Utils::DocumentView GetConfiguration() const { return m_configuration; }
  inline bool ConfigurationHasBeenSet() const { return m_configurationHasBeenSet; }
  template <typename ConfigurationT = Aws::Utils::Document>
  void SetConfiguration(ConfigurationT&& value) {
    m_configurationHasBeenSet = true;
    m_configuration = std::forward<ConfigurationT>(value);
  }
  template <typename ConfigurationT = Aws::Utils::Document>
  UpdateDataSourceRequest& WithConfiguration(ConfigurationT&& value) {
    SetConfiguration(std::forward<ConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const DataSourceVpcConfiguration& GetVpcConfiguration() const { return m_vpcConfiguration; }
  inline bool VpcConfigurationHasBeenSet() const { return m_vpcConfigurationHasBeenSet; }
  template <typename VpcConfigurationT = DataSourceVpcConfiguration>
  void SetVpcConfiguration(VpcConfigurationT&& value) {
    m_vpcConfigurationHasBeenSet = true;
    m_vpcConfiguration = std::forward<VpcConfigurationT>(value);
  }
  template <typename VpcConfigurationT = DataSourceVpcConfiguration>
  UpdateDataSourceRequest& WithVpcConfiguration(VpcConfigurationT&& value) {
    SetVpcConfiguration(std::forward<VpcConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the data source connector.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateDataSourceRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The chosen update frequency for your data source.</p>
   */
  inline const Aws::String& GetSyncSchedule() const { return m_syncSchedule; }
  inline bool SyncScheduleHasBeenSet() const { return m_syncScheduleHasBeenSet; }
  template <typename SyncScheduleT = Aws::String>
  void SetSyncSchedule(SyncScheduleT&& value) {
    m_syncScheduleHasBeenSet = true;
    m_syncSchedule = std::forward<SyncScheduleT>(value);
  }
  template <typename SyncScheduleT = Aws::String>
  UpdateDataSourceRequest& WithSyncSchedule(SyncScheduleT&& value) {
    SetSyncSchedule(std::forward<SyncScheduleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an IAM role with permission to access the
   * data source and required resources.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  UpdateDataSourceRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const DocumentEnrichmentConfiguration& GetDocumentEnrichmentConfiguration() const { return m_documentEnrichmentConfiguration; }
  inline bool DocumentEnrichmentConfigurationHasBeenSet() const { return m_documentEnrichmentConfigurationHasBeenSet; }
  template <typename DocumentEnrichmentConfigurationT = DocumentEnrichmentConfiguration>
  void SetDocumentEnrichmentConfiguration(DocumentEnrichmentConfigurationT&& value) {
    m_documentEnrichmentConfigurationHasBeenSet = true;
    m_documentEnrichmentConfiguration = std::forward<DocumentEnrichmentConfigurationT>(value);
  }
  template <typename DocumentEnrichmentConfigurationT = DocumentEnrichmentConfiguration>
  UpdateDataSourceRequest& WithDocumentEnrichmentConfiguration(DocumentEnrichmentConfigurationT&& value) {
    SetDocumentEnrichmentConfiguration(std::forward<DocumentEnrichmentConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration for extracting information from media in documents for your
   * data source.</p>
   */
  inline const MediaExtractionConfiguration& GetMediaExtractionConfiguration() const { return m_mediaExtractionConfiguration; }
  inline bool MediaExtractionConfigurationHasBeenSet() const { return m_mediaExtractionConfigurationHasBeenSet; }
  template <typename MediaExtractionConfigurationT = MediaExtractionConfiguration>
  void SetMediaExtractionConfiguration(MediaExtractionConfigurationT&& value) {
    m_mediaExtractionConfigurationHasBeenSet = true;
    m_mediaExtractionConfiguration = std::forward<MediaExtractionConfigurationT>(value);
  }
  template <typename MediaExtractionConfigurationT = MediaExtractionConfiguration>
  UpdateDataSourceRequest& WithMediaExtractionConfiguration(MediaExtractionConfigurationT&& value) {
    SetMediaExtractionConfiguration(std::forward<MediaExtractionConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationId;

  Aws::String m_indexId;

  Aws::String m_dataSourceId;

  Aws::String m_displayName;

  Aws::Utils::Document m_configuration;

  DataSourceVpcConfiguration m_vpcConfiguration;

  Aws::String m_description;

  Aws::String m_syncSchedule;

  Aws::String m_roleArn;

  DocumentEnrichmentConfiguration m_documentEnrichmentConfiguration;

  MediaExtractionConfiguration m_mediaExtractionConfiguration;
  bool m_applicationIdHasBeenSet = false;
  bool m_indexIdHasBeenSet = false;
  bool m_dataSourceIdHasBeenSet = false;
  bool m_displayNameHasBeenSet = false;
  bool m_configurationHasBeenSet = false;
  bool m_vpcConfigurationHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_syncScheduleHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_documentEnrichmentConfigurationHasBeenSet = false;
  bool m_mediaExtractionConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace QBusiness
}  // namespace Aws
