/*
 * Decompiled with CFR 0.152.
 */
package ghidra.framework.store.db;

import generic.jar.ResourceFile;
import ghidra.framework.Application;
import ghidra.framework.store.db.PackedDatabase;
import ghidra.framework.store.local.ItemDeserializer;
import ghidra.framework.store.local.LockFile;
import ghidra.util.Msg;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskMonitor;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.SyncFailedException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import utilities.util.FileUtilities;

public class PackedDatabaseCache {
    private static final String CACHE_DIR_PROPERTY = "pdb.cache.dir";
    private static final String CACHE_ENABLED_PROPERTY = "pdb.cache.enabled";
    private static final int SHELF_LIFE = 604800000;
    private static final String CACHE_DIR = "packed-db-cache";
    private static final String CACHE_MAP_FILE = "cache.map";
    private static final String MAP_SEPARATOR = ",";
    private static PackedDatabaseCache cache;
    private static final String PDB_PREFIX = "pdb";
    private static volatile boolean doCleanup;
    private static Boolean isEnabled;
    private final File cacheDir;
    private final File mapFile;
    private final LockFile lock;

    private PackedDatabaseCache(File cacheDir) throws IOException {
        this.cacheDir = cacheDir;
        if (PackedDatabaseCache.isEnabled()) {
            if (!cacheDir.mkdir() && !cacheDir.isDirectory()) {
                throw new IOException("Failed to create cache directory: " + String.valueOf(cacheDir));
            }
            if (!cacheDir.canExecute() || !cacheDir.canWrite()) {
                throw new IOException("permission denied: " + String.valueOf(cacheDir));
            }
            Msg.info((Object)this, (Object)("Packed database cache: " + String.valueOf(cacheDir)));
        } else {
            Msg.info((Object)this, (Object)"Packed database cache is disabled");
        }
        this.mapFile = new File(cacheDir, CACHE_MAP_FILE);
        this.lock = new LockFile(cacheDir, "cache", "u");
    }

    public static boolean isEnabled() {
        if (isEnabled == null) {
            isEnabled = true;
            String enabled = System.getProperty(CACHE_ENABLED_PROPERTY);
            if (enabled != null) {
                isEnabled = "true".equals(enabled = enabled.trim().toLowerCase()) || "yes".equals(enabled);
            }
        }
        return isEnabled;
    }

    public static synchronized PackedDatabaseCache getCache() throws IOException {
        if (cache == null) {
            File cacheDir = null;
            String dirpath = System.getProperty(CACHE_DIR_PROPERTY);
            cacheDir = dirpath != null ? new File(dirpath) : new File(Application.getUserCacheDirectory(), CACHE_DIR);
            cache = new PackedDatabaseCache(cacheDir);
        }
        return cache;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<CachedDB> readCache() throws IOException {
        ArrayList<CachedDB> list = new ArrayList<CachedDB>();
        if (!this.mapFile.exists()) {
            for (File f : this.cacheDir.listFiles()) {
                if (!f.isDirectory() || !f.getName().startsWith(PDB_PREFIX)) continue;
                FileUtilities.deleteDir((File)f);
            }
            return list;
        }
        boolean modified = false;
        long now = new Date().getTime();
        try (BufferedReader r = new BufferedReader(new FileReader(this.mapFile));){
            String line;
            while ((line = r.readLine()) != null) {
                long timeSinceLastAccess;
                if ((line = line.trim()).length() == 0) continue;
                CachedDB entry = new CachedDB(this, line);
                if (this.isBadDBDir(entry)) {
                    Msg.warn((Object)this, (Object)("Forcing removal of bad cached DB: " + entry.itemName + ", " + String.valueOf(entry.dbDir)));
                    entry.lastAccessTime = 0L;
                }
                if ((timeSinceLastAccess = now - entry.lastAccessTime) > 604800000L || !entry.dbDir.exists() || entry.refreshRequired() && !entry.originalPackedDBExists()) {
                    if (!doCleanup) continue;
                    FileUtilities.deleteDir((File)entry.dbDir);
                    modified = true;
                    continue;
                }
                list.add(entry);
            }
        }
        doCleanup = false;
        if (modified) {
            this.writeCacheList(list);
        }
        return list;
    }

    private boolean isBadDBDir(CachedDB entry) {
        File dbDir = entry.dbDir;
        File[] files = dbDir.listFiles();
        if (files == null) {
            Msg.debug((Object)this, (Object)("CachedDB directory not found: " + entry.itemName + ", " + String.valueOf(entry.dbDir)));
            return true;
        }
        if (files.length == 0) {
            entry.lastModifiedTime = 0L;
            if (!entry.originalPackedDBExists()) {
                Msg.debug((Object)this, (Object)("CachedDB has empty directory and packed file not found: " + entry.itemName + ", " + entry.packedDbFilePath));
                return true;
            }
            return false;
        }
        for (File f : files) {
            if (!f.getName().endsWith(".gbf") || f.length() == 0L) continue;
            return false;
        }
        Msg.debug((Object)this, (Object)("CachedDB is not empty but contains no *.gbf files: " + entry.itemName + ", " + entry.packedDbFilePath));
        return true;
    }

    private void writeCacheList(List<CachedDB> list) throws IOException {
        FileOutputStream out = new FileOutputStream(this.mapFile);
        PrintWriter w = new PrintWriter(out);
        for (CachedDB entry : list) {
            w.println(entry.getMapEntry());
        }
        try {
            out.getFD().sync();
        }
        catch (SyncFailedException syncFailedException) {
            // empty catch block
        }
        w.close();
    }

    private void addCacheMapEntry(CachedDB cachedDb) throws IOException {
        FileOutputStream out = new FileOutputStream(this.mapFile, true);
        PrintWriter w = new PrintWriter(out);
        w.println(cachedDb.getMapEntry());
        try {
            out.getFD().sync();
        }
        catch (SyncFailedException syncFailedException) {
            // empty catch block
        }
        w.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CachedDB createCachedDb(ResourceFile packedDbFile) throws IOException {
        File dbDir = this.createCachedDir();
        ItemDeserializer itemDeserializer = null;
        boolean success = false;
        try {
            if (!packedDbFile.isFile()) {
                throw new FileNotFoundException("File not found: " + String.valueOf(packedDbFile));
            }
            itemDeserializer = new ItemDeserializer(packedDbFile);
            if (itemDeserializer.getFileType() != 0) {
                throw new IOException("Incorrect file type");
            }
            String contentType = itemDeserializer.getContentType();
            String itemName = itemDeserializer.getItemName();
            CachedDB cachedDb = new CachedDB(this, packedDbFile, dbDir, contentType, itemName, 0L, true);
            success = true;
            CachedDB cachedDB = cachedDb;
            return cachedDB;
        }
        finally {
            if (itemDeserializer != null) {
                itemDeserializer.dispose();
            }
            if (!success) {
                FileUtilities.deleteDir((File)dbDir);
            }
        }
    }

    private File createCachedDir() throws IOException {
        int tries = 0;
        while (tries++ < 10) {
            File dir = new File(this.cacheDir, PDB_PREFIX + PackedDatabase.getRandomString());
            if (!dir.mkdir()) continue;
            return dir;
        }
        throw new IOException("Unable to create cached database");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    CachedDB getCachedDBEntry(ResourceFile packedDbFile) throws IOException {
        if (!PackedDatabaseCache.isEnabled()) {
            throw new IOException("Cache disabled");
        }
        if (!this.lock.createLock(30000, true)) {
            throw new IOException("Packed database cache timeout");
        }
        try {
            String packedFilePath = packedDbFile.getCanonicalPath();
            List<CachedDB> list = this.readCache();
            for (CachedDB entry : list) {
                if (!packedFilePath.equals(entry.packedDbFilePath)) continue;
                CachedDB cachedDB = entry;
                return cachedDB;
            }
        }
        finally {
            this.lock.removeLock();
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void purgeFromCache(ResourceFile packedDbFile) throws IOException {
        if (!this.lock.createLock(30000, true)) {
            throw new IOException("Packed database cache timeout");
        }
        try {
            String packedFilePath = packedDbFile.getCanonicalPath();
            List<CachedDB> list = this.readCache();
            for (CachedDB entry : list) {
                if (!packedFilePath.equals(entry.packedDbFilePath)) continue;
                entry.lastAccessTime = 0L;
                this.writeCacheList(list);
                FileUtilities.deleteDir((File)entry.dbDir);
                break;
            }
        }
        finally {
            this.lock.removeLock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isInCache(ResourceFile packedDbFile) throws IOException {
        if (!this.lock.createLock(30000, true)) {
            throw new IOException("Packed database cache timeout");
        }
        try {
            String packedFilePath = packedDbFile.getCanonicalPath();
            List<CachedDB> list = this.readCache();
            for (CachedDB entry : list) {
                if (!packedFilePath.equals(entry.packedDbFilePath) || !entry.dbDir.isDirectory()) continue;
                boolean bl = true;
                return bl;
            }
        }
        finally {
            this.lock.removeLock();
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    PackedDatabase getCachedDB(ResourceFile packedDbFile, TaskMonitor monitor) throws CancelledException, IOException {
        if (!PackedDatabaseCache.isEnabled()) {
            throw new IOException("Cache disabled");
        }
        if (!this.lock.createLock(30000, true)) {
            throw new IOException("Packed database cache timeout");
        }
        try {
            boolean isReadOnly = PackedDatabase.isReadOnlyPDBDirectory(packedDbFile.getParentFile());
            LockFile packedDbLock = null;
            if (!isReadOnly) {
                packedDbLock = PackedDatabase.getFileLock(packedDbFile.getFile(false));
                PackedDatabase.lock(packedDbLock, true, true);
            }
            try {
                String packedFilePath = packedDbFile.getCanonicalPath();
                long now = new Date().getTime();
                CachedDB cachedDb = null;
                List<CachedDB> list = this.readCache();
                for (CachedDB entry : list) {
                    if (!packedFilePath.equals(entry.packedDbFilePath)) continue;
                    if (!entry.dbDir.canExecute() || !entry.dbDir.canWrite()) {
                        throw new IOException("Permssion denied: " + String.valueOf(entry.dbDir));
                    }
                    entry.lastAccessTime = now;
                    this.writeCacheList(list);
                    cachedDb = entry;
                    break;
                }
                if (cachedDb == null) {
                    cachedDb = this.createCachedDb(packedDbFile);
                    cachedDb.lastAccessTime = now;
                    this.addCacheMapEntry(cachedDb);
                    Msg.debug((Object)this, (Object)("Caching packed database: " + cachedDb.packedDbFilePath));
                } else {
                    Msg.debug((Object)this, (Object)("Using cached packed database: " + cachedDb.packedDbFilePath));
                }
                Object object = new PackedDatabase(cachedDb, packedDbFile, packedDbLock, monitor);
                if (packedDbLock != null && packedDbLock.haveLock()) {
                    packedDbLock.removeLock();
                }
                return object;
            }
            catch (Throwable throwable) {
                if (packedDbLock != null && packedDbLock.haveLock()) {
                    packedDbLock.removeLock();
                }
                throw throwable;
            }
        }
        finally {
            this.lock.removeLock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void updateLastModified(ResourceFile packedDbFile, long modTime) throws IOException {
        if (!PackedDatabaseCache.isEnabled()) {
            throw new IOException("Cache disabled");
        }
        if (!this.lock.createLock(30000, true)) {
            throw new IOException("Packed database cache timeout");
        }
        try {
            String packedFilePath = packedDbFile.getCanonicalPath();
            long now = new Date().getTime();
            List<CachedDB> list = this.readCache();
            for (CachedDB entry : list) {
                if (!packedFilePath.equals(entry.packedDbFilePath)) continue;
                entry.lastAccessTime = now;
                entry.lastModifiedTime = modTime;
                this.writeCacheList(list);
                Msg.debug((Object)this, (Object)("Cache update completed: " + packedFilePath));
                return;
            }
            Msg.debug((Object)this, (Object)("Cache entry not found for: " + packedFilePath));
        }
        finally {
            this.lock.removeLock();
        }
    }

    static {
        doCleanup = true;
    }

    class CachedDB {
        public final String packedDbFilePath;
        public final String itemName;
        public final String contentType;
        public final File dbDir;
        private ResourceFile packedDbFile;
        private boolean refreshRequired;
        private long lastModifiedTime;
        private long lastAccessTime;

        CachedDB(PackedDatabaseCache this$0, ResourceFile packedDbFile, File dbDir, String contentType, String itemName, long lastModifiedTime, boolean refreshRequired) throws IOException {
            this.packedDbFile = packedDbFile;
            this.packedDbFilePath = packedDbFile.getCanonicalPath();
            this.dbDir = dbDir;
            this.contentType = contentType;
            this.itemName = itemName;
            this.lastModifiedTime = lastModifiedTime;
            this.refreshRequired = refreshRequired;
        }

        CachedDB(PackedDatabaseCache this$0, String mapEntry) {
            String[] split = this.splitEntry(mapEntry);
            this.packedDbFilePath = split[0];
            this.dbDir = new File(this$0.cacheDir, split[1]);
            this.lastModifiedTime = Long.parseUnsignedLong(split[2], 16);
            this.contentType = split[3];
            this.itemName = split[4];
            try {
                this.packedDbFile = new ResourceFile(this.packedDbFilePath);
                this.refreshRequired = this.lastModifiedTime != this.packedDbFile.lastModified();
            }
            catch (Exception exception) {
                // empty catch block
            }
            this.lastAccessTime = Long.parseUnsignedLong(split[5], 16);
        }

        String[] splitEntry(String mapEntry) {
            String[] split = new String[6];
            int lastIndex = mapEntry.length();
            for (int i = 5; i > 0; --i) {
                int index = mapEntry.lastIndexOf(PackedDatabaseCache.MAP_SEPARATOR, lastIndex - 1);
                if (index <= 0 || lastIndex - index == 1) {
                    throw new IllegalArgumentException("Invalid cache map entry");
                }
                split[i] = mapEntry.substring(index + 1, lastIndex);
                lastIndex = index;
            }
            split[0] = mapEntry.substring(0, lastIndex);
            return split;
        }

        boolean refreshRequired() {
            return this.refreshRequired;
        }

        long getLastModified() {
            return this.lastModifiedTime;
        }

        boolean originalPackedDBExists() {
            return this.packedDbFile != null && this.packedDbFile.isFile();
        }

        String getMapEntry() {
            StringBuffer buf = new StringBuffer();
            buf.append(this.packedDbFilePath);
            buf.append(PackedDatabaseCache.MAP_SEPARATOR);
            buf.append(this.dbDir.getName());
            buf.append(PackedDatabaseCache.MAP_SEPARATOR);
            buf.append(Long.toHexString(this.lastModifiedTime));
            buf.append(PackedDatabaseCache.MAP_SEPARATOR);
            buf.append(this.contentType);
            buf.append(PackedDatabaseCache.MAP_SEPARATOR);
            buf.append(this.itemName);
            buf.append(PackedDatabaseCache.MAP_SEPARATOR);
            buf.append(Long.toHexString(this.lastAccessTime));
            return buf.toString();
        }
    }
}

