#!/usr/bin/env perl

use strict;
use warnings;

use Amazon::Credentials;

use Amazon::API::Lambda;
use Amazon::API::EC2;
use Amazon::API::SecretsManager;

use Text::ASCIITable::EasyTable;
use JSON::PP;

use Data::Dumper;

use constant {
  TRUE  => 1,
  FALSE => 0,
};

########################################################################
sub format_tags {
########################################################################
  my ($data) = @_;

  foreach my $t ( @{$data} ) {
    my $tags = $t->{Tags};

    $t->{TagText} = join "\n", grep { !/^=/xsm }
      map { sprintf '%s=%s', @{$_}{qw(Key Value)} } @{$tags};
  }

  return $data;
}

my $index;

########################################################################
sub easytable_get_row {
########################################################################
  my ( $data, $key, $index ) = @_;

  return $data->{ $index->{$key} };
}

########################################################################
sub list_secrets {
########################################################################
  my ($credentials) = @_;

  my $secrets
    = Amazon::API::SecretsManager->new( credentials => $credentials );

  my $secret_list = $secrets->ListSecrets;

  print {*STDOUT} JSON::PP->new->pretty->encode($secret_list);

  format_tags( $secret_list->{SecretList} );

  $index = [
    reverse TagText  => 'Tags',
    LastAccessedDate => 'Last Access',
    CreatedDate      => 'Date Created',
    Description      => 'Description',
    Name             => 'Name',
  ];

  print easy_table(
    table_options => { headingText => 'Secrets' },
    data          => $secret_list->{SecretList},
    index         => $index,
  );

  return @{ $secret_list->{SecretList} };
}

########################################################################
sub list_subnets {
########################################################################
  my ($credentials) = @_;

  my $ec2 = Amazon::API::EC2->new(
    { region      => 'us-east-1',
      credentials => $credentials,
    }
  );

  my $subnets = $ec2->DescribeSubnets();

  print {*STDOUT} JSON::PP->new->pretty->encode($subnets);

  format_tags($subnets);

  my @sorted_subnets
    = sort { $a->{VpcId} cmp $b->{VpcId} or $a->{TagText} cmp $b->{TagText} }
    @{$subnets};

  $index = [
    reverse TagText  => 'Tags',
    AvailabilityZone => 'AZ',
    VpcId            => 'VPC',
    CidrBlock        => 'CIDR',
    SubnetId         => 'Subnet',
  ];

  print easy_table(
    table_options => { headingText => 'Subnets' },
    data          => \@sorted_subnets,
    index         => $index,
  );

  return @{$subnets};
}

########################################################################
sub list_layers {
########################################################################
  my ($credentials) = @_;

  my $lambda = Amazon::API::Lambda->new(
    { region      => 'us-east-1',
      credentials => $credentials,
    }
  );

  my $layers = $lambda->ListLayers();

  return print {*STDOUT} JSON::PP->new->pretty->encode($layers);
}

########################################################################
sub main {
########################################################################

  my $credentials
    = Amazon::Credentials->new( { no_passkey_warning => TRUE } );

  list_layers($credentials);

  # list_secrets($credentials);

  # list_subnets($credentials);

  return TRUE;
}

main();

1;
