use strict;
use Test::More;
use Test::Number::Delta;
use Vector::QRCode::EPS;
use t::EPS::Util;

can_ok 'Vector::QRCode::EPS', 'generate';

my $data = Vector::QRCode::EPS->generate(
    text => 'This is a test',
    qrcode_options => {
        margin  => 4,
        version => 5,
        levelt  => 'H',
    },
);
my $user = getlogin || 'Console';

isa_ok $data, 'PostScript::Simple';

my $result = $data->get;

my ($creation_date) = $result =~ /CreationDate: (.+)\n/;

my $expect = do {local $/; <DATA>};
$expect =~ s/__TIME__/$creation_date/;
$expect =~ s/__USER__/$user/;

my $res = t::EPS::Util->new($result);
my $exp = t::EPS::Util->new($expect);

my @res_body = $res->body;
my @exp_body = $exp->body;

my @res_qr_cell = splice(@res_body, 3);
my @exp_qr_cell = splice(@exp_body, 3);

subtest cell_accuracy_test => sub {

    is scalar(@res_qr_cell), scalar(@exp_qr_cell), sprintf('%d cells defination is there', scalar(@exp_qr_cell));

    my $cell_pattern = qr/^([0-9\.]+) ucm ([0-9\.]+) ucm ([0-9\.]+) ucm ([0-9\.]+) ucm box fill$/;

    for my $i (0 .. $#exp_qr_cell) {
        my @res_cell_def = $res_qr_cell[$i] =~ $cell_pattern;
        my @exp_cell_def = $exp_qr_cell[$i] =~ $cell_pattern;
    
        delta_within(
            \@res_cell_def, \@exp_cell_def, 1e-12,
            sprintf("compare: \n==RES==\n%s\n==EXPECT==\n%s\n", $res_qr_cell[$i], $exp_qr_cell[$i])
        );
    }
};

done_testing;

__DATA__
%!PS-Adobe-3.0 EPSF-1.2
%%Title: (PostScript::Simple generated page)
%%LanguageLevel: 1
%%Creator: PostScript::Simple perl module version 0.08
%%CreationDate: __TIME__
%%For: __USER__
%%Orientation: Portrait
%%BoundingBox: 0 0 283 283
%%EndComments
%%BeginProlog
/ll 1 def systemdict /languagelevel known {
/ll languagelevel def } if
%%BeginResource: PostScript::Simple
/STARTDIFFENC { mark } bind def
/ENDDIFFENC { 

% /NewEnc BaseEnc STARTDIFFENC number or glyphname ... ENDDIFFENC -
	counttomark 2 add -1 roll 256 array copy
	/TempEncode exch def

	% pointer for sequential encodings
	/EncodePointer 0 def
	{
		% Get the bottom object
		counttomark -1 roll
		% Is it a mark?
		dup type dup /marktype eq {
			% End of encoding
			pop pop exit
		} {
			/nametype eq {
			% Insert the name at EncodePointer 

			% and increment the pointer.
			TempEncode EncodePointer 3 -1 roll put
			/EncodePointer EncodePointer 1 add def
			} {
			% Set the EncodePointer to the number
			/EncodePointer exch def
			} ifelse
		} ifelse
	} loop

	TempEncode def
} bind def

% Define ISO Latin1 encoding if it doesnt exist
/ISOLatin1Encoding where {
%	(ISOLatin1 exists!) =
	pop
} {
	(ISOLatin1 does not exist, creating...) =
	/ISOLatin1Encoding StandardEncoding STARTDIFFENC
		144 /dotlessi /grave /acute /circumflex /tilde 
		/macron /breve /dotaccent /dieresis /.notdef /ring 
		/cedilla /.notdef /hungarumlaut /ogonek /caron /space 
		/exclamdown /cent /sterling /currency /yen /brokenbar 
		/section /dieresis /copyright /ordfeminine 
		/guillemotleft /logicalnot /hyphen /registered 
		/macron /degree /plusminus /twosuperior 
		/threesuperior /acute /mu /paragraph /periodcentered 
		/cedilla /onesuperior /ordmasculine /guillemotright 
		/onequarter /onehalf /threequarters /questiondown 
		/Agrave /Aacute /Acircumflex /Atilde /Adieresis 
		/Aring /AE /Ccedilla /Egrave /Eacute /Ecircumflex 
		/Edieresis /Igrave /Iacute /Icircumflex /Idieresis 
		/Eth /Ntilde /Ograve /Oacute /Ocircumflex /Otilde 
		/Odieresis /multiply /Oslash /Ugrave /Uacute 
		/Ucircumflex /Udieresis /Yacute /Thorn /germandbls 
		/agrave /aacute /acircumflex /atilde /adieresis 
		/aring /ae /ccedilla /egrave /eacute /ecircumflex 
		/edieresis /igrave /iacute /icircumflex /idieresis 
		/eth /ntilde /ograve /oacute /ocircumflex /otilde 
		/odieresis /divide /oslash /ugrave /uacute 
		/ucircumflex /udieresis /yacute /thorn /ydieresis
	ENDDIFFENC
} ifelse

% Name: Re-encode Font
% Description: Creates a new font using the named encoding. 

/REENCODEFONT { % /Newfont NewEncoding /Oldfont
	findfont dup length 4 add dict
	begin
		{ % forall
			1 index /FID ne 
			2 index /UniqueID ne and
			2 index /XUID ne and
			{ def } { pop pop } ifelse
		} forall
		/Encoding exch def
		% defs for DPS
		/BitmapWidths false def
		/ExactSize 0 def
		/InBetweenSize 0 def
		/TransformedChar 0 def
		currentdict
	end
	definefont pop
} bind def

% Reencode the std fonts: 
/Courier-iso ISOLatin1Encoding /Courier REENCODEFONT
/Courier-Bold-iso ISOLatin1Encoding /Courier-Bold REENCODEFONT
/Courier-BoldOblique-iso ISOLatin1Encoding /Courier-BoldOblique REENCODEFONT
/Courier-Oblique-iso ISOLatin1Encoding /Courier-Oblique REENCODEFONT
/Helvetica-iso ISOLatin1Encoding /Helvetica REENCODEFONT
/Helvetica-Bold-iso ISOLatin1Encoding /Helvetica-Bold REENCODEFONT
/Helvetica-BoldOblique-iso ISOLatin1Encoding /Helvetica-BoldOblique REENCODEFONT
/Helvetica-Oblique-iso ISOLatin1Encoding /Helvetica-Oblique REENCODEFONT
/Times-Roman-iso ISOLatin1Encoding /Times-Roman REENCODEFONT
/Times-Bold-iso ISOLatin1Encoding /Times-Bold REENCODEFONT
/Times-BoldItalic-iso ISOLatin1Encoding /Times-BoldItalic REENCODEFONT
/Times-Italic-iso ISOLatin1Encoding /Times-Italic REENCODEFONT
/Symbol-iso ISOLatin1Encoding /Symbol REENCODEFONT
/box {
  newpath 3 copy pop exch 4 copy pop pop
  8 copy pop pop pop pop exch pop exch
  3 copy pop pop exch moveto lineto
  lineto lineto pop pop pop pop closepath
} bind def
/ucm {72 mul 2.54 div} def
%%EndResource
%%EndProlog
1 1 1 setrgbcolor
0 ucm 0 ucm 10 ucm 10 ucm box fill
0.03922 0.03922 0.03922 setrgbcolor
0 ucm 0 ucm 0.27027027027027 ucm 0.27027027027027 ucm box fill
0.27027027027027 ucm 0 ucm 0.540540540540541 ucm 0.27027027027027 ucm box fill
0.540540540540541 ucm 0 ucm 0.810810810810811 ucm 0.27027027027027 ucm box fill
0.810810810810811 ucm 0 ucm 1.08108108108108 ucm 0.27027027027027 ucm box fill
1.08108108108108 ucm 0 ucm 1.35135135135135 ucm 0.27027027027027 ucm box fill
1.35135135135135 ucm 0 ucm 1.62162162162162 ucm 0.27027027027027 ucm box fill
1.62162162162162 ucm 0 ucm 1.89189189189189 ucm 0.27027027027027 ucm box fill
2.16216216216216 ucm 0 ucm 2.43243243243243 ucm 0.27027027027027 ucm box fill
2.97297297297297 ucm 0 ucm 3.24324324324324 ucm 0.27027027027027 ucm box fill
3.24324324324324 ucm 0 ucm 3.51351351351351 ucm 0.27027027027027 ucm box fill
3.51351351351351 ucm 0 ucm 3.78378378378378 ucm 0.27027027027027 ucm box fill
4.05405405405405 ucm 0 ucm 4.32432432432432 ucm 0.27027027027027 ucm box fill
4.32432432432432 ucm 0 ucm 4.59459459459459 ucm 0.27027027027027 ucm box fill
4.59459459459459 ucm 0 ucm 4.86486486486486 ucm 0.27027027027027 ucm box fill
5.13513513513514 ucm 0 ucm 5.40540540540541 ucm 0.27027027027027 ucm box fill
5.40540540540541 ucm 0 ucm 5.67567567567568 ucm 0.27027027027027 ucm box fill
5.67567567567568 ucm 0 ucm 5.94594594594595 ucm 0.27027027027027 ucm box fill
6.21621621621622 ucm 0 ucm 6.48648648648649 ucm 0.27027027027027 ucm box fill
6.48648648648649 ucm 0 ucm 6.75675675675676 ucm 0.27027027027027 ucm box fill
6.75675675675676 ucm 0 ucm 7.02702702702703 ucm 0.27027027027027 ucm box fill
7.2972972972973 ucm 0 ucm 7.56756756756757 ucm 0.27027027027027 ucm box fill
7.56756756756757 ucm 0 ucm 7.83783783783784 ucm 0.27027027027027 ucm box fill
8.10810810810811 ucm 0 ucm 8.37837837837838 ucm 0.27027027027027 ucm box fill
8.37837837837838 ucm 0 ucm 8.64864864864865 ucm 0.27027027027027 ucm box fill
8.64864864864865 ucm 0 ucm 8.91891891891892 ucm 0.27027027027027 ucm box fill
9.72972972972973 ucm 0 ucm 10 ucm 0.27027027027027 ucm box fill
0 ucm 0.27027027027027 ucm 0.27027027027027 ucm 0.540540540540541 ucm box fill
1.62162162162162 ucm 0.27027027027027 ucm 1.89189189189189 ucm 0.540540540540541 ucm box fill
2.16216216216216 ucm 0.27027027027027 ucm 2.43243243243243 ucm 0.540540540540541 ucm box fill
2.43243243243243 ucm 0.27027027027027 ucm 2.7027027027027 ucm 0.540540540540541 ucm box fill
2.7027027027027 ucm 0.27027027027027 ucm 2.97297297297297 ucm 0.540540540540541 ucm box fill
3.51351351351351 ucm 0.27027027027027 ucm 3.78378378378378 ucm 0.540540540540541 ucm box fill
4.59459459459459 ucm 0.27027027027027 ucm 4.86486486486486 ucm 0.540540540540541 ucm box fill
5.67567567567568 ucm 0.27027027027027 ucm 5.94594594594595 ucm 0.540540540540541 ucm box fill
6.75675675675676 ucm 0.27027027027027 ucm 7.02702702702703 ucm 0.540540540540541 ucm box fill
7.2972972972973 ucm 0.27027027027027 ucm 7.56756756756757 ucm 0.540540540540541 ucm box fill
8.37837837837838 ucm 0.27027027027027 ucm 8.64864864864865 ucm 0.540540540540541 ucm box fill
8.64864864864865 ucm 0.27027027027027 ucm 8.91891891891892 ucm 0.540540540540541 ucm box fill
8.91891891891892 ucm 0.27027027027027 ucm 9.18918918918919 ucm 0.540540540540541 ucm box fill
0 ucm 0.540540540540541 ucm 0.27027027027027 ucm 0.810810810810811 ucm box fill
0.540540540540541 ucm 0.540540540540541 ucm 0.810810810810811 ucm 0.810810810810811 ucm box fill
0.810810810810811 ucm 0.540540540540541 ucm 1.08108108108108 ucm 0.810810810810811 ucm box fill
1.08108108108108 ucm 0.540540540540541 ucm 1.35135135135135 ucm 0.810810810810811 ucm box fill
1.62162162162162 ucm 0.540540540540541 ucm 1.89189189189189 ucm 0.810810810810811 ucm box fill
2.16216216216216 ucm 0.540540540540541 ucm 2.43243243243243 ucm 0.810810810810811 ucm box fill
2.43243243243243 ucm 0.540540540540541 ucm 2.7027027027027 ucm 0.810810810810811 ucm box fill
2.7027027027027 ucm 0.540540540540541 ucm 2.97297297297297 ucm 0.810810810810811 ucm box fill
2.97297297297297 ucm 0.540540540540541 ucm 3.24324324324324 ucm 0.810810810810811 ucm box fill
4.05405405405405 ucm 0.540540540540541 ucm 4.32432432432432 ucm 0.810810810810811 ucm box fill
5.13513513513514 ucm 0.540540540540541 ucm 5.40540540540541 ucm 0.810810810810811 ucm box fill
6.21621621621622 ucm 0.540540540540541 ucm 6.48648648648649 ucm 0.810810810810811 ucm box fill
7.83783783783784 ucm 0.540540540540541 ucm 8.10810810810811 ucm 0.810810810810811 ucm box fill
8.10810810810811 ucm 0.540540540540541 ucm 8.37837837837838 ucm 0.810810810810811 ucm box fill
8.64864864864865 ucm 0.540540540540541 ucm 8.91891891891892 ucm 0.810810810810811 ucm box fill
9.18918918918919 ucm 0.540540540540541 ucm 9.45945945945946 ucm 0.810810810810811 ucm box fill
9.45945945945946 ucm 0.540540540540541 ucm 9.72972972972973 ucm 0.810810810810811 ucm box fill
9.72972972972973 ucm 0.540540540540541 ucm 10 ucm 0.810810810810811 ucm box fill
0 ucm 0.810810810810811 ucm 0.27027027027027 ucm 1.08108108108108 ucm box fill
0.540540540540541 ucm 0.810810810810811 ucm 0.810810810810811 ucm 1.08108108108108 ucm box fill
0.810810810810811 ucm 0.810810810810811 ucm 1.08108108108108 ucm 1.08108108108108 ucm box fill
1.08108108108108 ucm 0.810810810810811 ucm 1.35135135135135 ucm 1.08108108108108 ucm box fill
1.62162162162162 ucm 0.810810810810811 ucm 1.89189189189189 ucm 1.08108108108108 ucm box fill
3.51351351351351 ucm 0.810810810810811 ucm 3.78378378378378 ucm 1.08108108108108 ucm box fill
3.78378378378378 ucm 0.810810810810811 ucm 4.05405405405405 ucm 1.08108108108108 ucm box fill
4.59459459459459 ucm 0.810810810810811 ucm 4.86486486486486 ucm 1.08108108108108 ucm box fill
4.86486486486486 ucm 0.810810810810811 ucm 5.13513513513514 ucm 1.08108108108108 ucm box fill
5.67567567567568 ucm 0.810810810810811 ucm 5.94594594594595 ucm 1.08108108108108 ucm box fill
5.94594594594595 ucm 0.810810810810811 ucm 6.21621621621622 ucm 1.08108108108108 ucm box fill
6.75675675675676 ucm 0.810810810810811 ucm 7.02702702702703 ucm 1.08108108108108 ucm box fill
7.02702702702703 ucm 0.810810810810811 ucm 7.2972972972973 ucm 1.08108108108108 ucm box fill
7.56756756756757 ucm 0.810810810810811 ucm 7.83783783783784 ucm 1.08108108108108 ucm box fill
7.83783783783784 ucm 0.810810810810811 ucm 8.10810810810811 ucm 1.08108108108108 ucm box fill
8.10810810810811 ucm 0.810810810810811 ucm 8.37837837837838 ucm 1.08108108108108 ucm box fill
8.37837837837838 ucm 0.810810810810811 ucm 8.64864864864865 ucm 1.08108108108108 ucm box fill
8.64864864864865 ucm 0.810810810810811 ucm 8.91891891891892 ucm 1.08108108108108 ucm box fill
8.91891891891892 ucm 0.810810810810811 ucm 9.18918918918919 ucm 1.08108108108108 ucm box fill
0 ucm 1.08108108108108 ucm 0.27027027027027 ucm 1.35135135135135 ucm box fill
0.540540540540541 ucm 1.08108108108108 ucm 0.810810810810811 ucm 1.35135135135135 ucm box fill
0.810810810810811 ucm 1.08108108108108 ucm 1.08108108108108 ucm 1.35135135135135 ucm box fill
1.08108108108108 ucm 1.08108108108108 ucm 1.35135135135135 ucm 1.35135135135135 ucm box fill
1.62162162162162 ucm 1.08108108108108 ucm 1.89189189189189 ucm 1.35135135135135 ucm box fill
2.43243243243243 ucm 1.08108108108108 ucm 2.7027027027027 ucm 1.35135135135135 ucm box fill
2.7027027027027 ucm 1.08108108108108 ucm 2.97297297297297 ucm 1.35135135135135 ucm box fill
3.51351351351351 ucm 1.08108108108108 ucm 3.78378378378378 ucm 1.35135135135135 ucm box fill
3.78378378378378 ucm 1.08108108108108 ucm 4.05405405405405 ucm 1.35135135135135 ucm box fill
4.59459459459459 ucm 1.08108108108108 ucm 4.86486486486486 ucm 1.35135135135135 ucm box fill
4.86486486486486 ucm 1.08108108108108 ucm 5.13513513513514 ucm 1.35135135135135 ucm box fill
5.67567567567568 ucm 1.08108108108108 ucm 5.94594594594595 ucm 1.35135135135135 ucm box fill
5.94594594594595 ucm 1.08108108108108 ucm 6.21621621621622 ucm 1.35135135135135 ucm box fill
6.75675675675676 ucm 1.08108108108108 ucm 7.02702702702703 ucm 1.35135135135135 ucm box fill
7.02702702702703 ucm 1.08108108108108 ucm 7.2972972972973 ucm 1.35135135135135 ucm box fill
7.56756756756757 ucm 1.08108108108108 ucm 7.83783783783784 ucm 1.35135135135135 ucm box fill
7.83783783783784 ucm 1.08108108108108 ucm 8.10810810810811 ucm 1.35135135135135 ucm box fill
8.10810810810811 ucm 1.08108108108108 ucm 8.37837837837838 ucm 1.35135135135135 ucm box fill
8.37837837837838 ucm 1.08108108108108 ucm 8.64864864864865 ucm 1.35135135135135 ucm box fill
8.64864864864865 ucm 1.08108108108108 ucm 8.91891891891892 ucm 1.35135135135135 ucm box fill
8.91891891891892 ucm 1.08108108108108 ucm 9.18918918918919 ucm 1.35135135135135 ucm box fill
0 ucm 1.35135135135135 ucm 0.27027027027027 ucm 1.62162162162162 ucm box fill
1.62162162162162 ucm 1.35135135135135 ucm 1.89189189189189 ucm 1.62162162162162 ucm box fill
2.16216216216216 ucm 1.35135135135135 ucm 2.43243243243243 ucm 1.62162162162162 ucm box fill
2.43243243243243 ucm 1.35135135135135 ucm 2.7027027027027 ucm 1.62162162162162 ucm box fill
2.7027027027027 ucm 1.35135135135135 ucm 2.97297297297297 ucm 1.62162162162162 ucm box fill
2.97297297297297 ucm 1.35135135135135 ucm 3.24324324324324 ucm 1.62162162162162 ucm box fill
3.24324324324324 ucm 1.35135135135135 ucm 3.51351351351351 ucm 1.62162162162162 ucm box fill
3.78378378378378 ucm 1.35135135135135 ucm 4.05405405405405 ucm 1.62162162162162 ucm box fill
4.05405405405405 ucm 1.35135135135135 ucm 4.32432432432432 ucm 1.62162162162162 ucm box fill
4.32432432432432 ucm 1.35135135135135 ucm 4.59459459459459 ucm 1.62162162162162 ucm box fill
4.86486486486486 ucm 1.35135135135135 ucm 5.13513513513514 ucm 1.62162162162162 ucm box fill
5.13513513513514 ucm 1.35135135135135 ucm 5.40540540540541 ucm 1.62162162162162 ucm box fill
5.40540540540541 ucm 1.35135135135135 ucm 5.67567567567568 ucm 1.62162162162162 ucm box fill
5.94594594594595 ucm 1.35135135135135 ucm 6.21621621621622 ucm 1.62162162162162 ucm box fill
6.21621621621622 ucm 1.35135135135135 ucm 6.48648648648649 ucm 1.62162162162162 ucm box fill
6.48648648648649 ucm 1.35135135135135 ucm 6.75675675675676 ucm 1.62162162162162 ucm box fill
7.02702702702703 ucm 1.35135135135135 ucm 7.2972972972973 ucm 1.62162162162162 ucm box fill
7.2972972972973 ucm 1.35135135135135 ucm 7.56756756756757 ucm 1.62162162162162 ucm box fill
7.56756756756757 ucm 1.35135135135135 ucm 7.83783783783784 ucm 1.62162162162162 ucm box fill
8.64864864864865 ucm 1.35135135135135 ucm 8.91891891891892 ucm 1.62162162162162 ucm box fill
8.91891891891892 ucm 1.35135135135135 ucm 9.18918918918919 ucm 1.62162162162162 ucm box fill
9.72972972972973 ucm 1.35135135135135 ucm 10 ucm 1.62162162162162 ucm box fill
0 ucm 1.62162162162162 ucm 0.27027027027027 ucm 1.89189189189189 ucm box fill
0.27027027027027 ucm 1.62162162162162 ucm 0.540540540540541 ucm 1.89189189189189 ucm box fill
0.540540540540541 ucm 1.62162162162162 ucm 0.810810810810811 ucm 1.89189189189189 ucm box fill
0.810810810810811 ucm 1.62162162162162 ucm 1.08108108108108 ucm 1.89189189189189 ucm box fill
1.08108108108108 ucm 1.62162162162162 ucm 1.35135135135135 ucm 1.89189189189189 ucm box fill
1.35135135135135 ucm 1.62162162162162 ucm 1.62162162162162 ucm 1.89189189189189 ucm box fill
1.62162162162162 ucm 1.62162162162162 ucm 1.89189189189189 ucm 1.89189189189189 ucm box fill
3.24324324324324 ucm 1.62162162162162 ucm 3.51351351351351 ucm 1.89189189189189 ucm box fill
3.51351351351351 ucm 1.62162162162162 ucm 3.78378378378378 ucm 1.89189189189189 ucm box fill
3.78378378378378 ucm 1.62162162162162 ucm 4.05405405405405 ucm 1.89189189189189 ucm box fill
4.32432432432432 ucm 1.62162162162162 ucm 4.59459459459459 ucm 1.89189189189189 ucm box fill
4.59459459459459 ucm 1.62162162162162 ucm 4.86486486486486 ucm 1.89189189189189 ucm box fill
4.86486486486486 ucm 1.62162162162162 ucm 5.13513513513514 ucm 1.89189189189189 ucm box fill
5.40540540540541 ucm 1.62162162162162 ucm 5.67567567567568 ucm 1.89189189189189 ucm box fill
5.67567567567568 ucm 1.62162162162162 ucm 5.94594594594595 ucm 1.89189189189189 ucm box fill
5.94594594594595 ucm 1.62162162162162 ucm 6.21621621621622 ucm 1.89189189189189 ucm box fill
6.48648648648649 ucm 1.62162162162162 ucm 6.75675675675676 ucm 1.89189189189189 ucm box fill
6.75675675675676 ucm 1.62162162162162 ucm 7.02702702702703 ucm 1.89189189189189 ucm box fill
7.02702702702703 ucm 1.62162162162162 ucm 7.2972972972973 ucm 1.89189189189189 ucm box fill
7.2972972972973 ucm 1.62162162162162 ucm 7.56756756756757 ucm 1.89189189189189 ucm box fill
7.56756756756757 ucm 1.62162162162162 ucm 7.83783783783784 ucm 1.89189189189189 ucm box fill
8.10810810810811 ucm 1.62162162162162 ucm 8.37837837837838 ucm 1.89189189189189 ucm box fill
8.64864864864865 ucm 1.62162162162162 ucm 8.91891891891892 ucm 1.89189189189189 ucm box fill
8.91891891891892 ucm 1.62162162162162 ucm 9.18918918918919 ucm 1.89189189189189 ucm box fill
9.18918918918919 ucm 1.62162162162162 ucm 9.45945945945946 ucm 1.89189189189189 ucm box fill
9.72972972972973 ucm 1.62162162162162 ucm 10 ucm 1.89189189189189 ucm box fill
2.16216216216216 ucm 1.89189189189189 ucm 2.43243243243243 ucm 2.16216216216216 ucm box fill
2.43243243243243 ucm 1.89189189189189 ucm 2.7027027027027 ucm 2.16216216216216 ucm box fill
3.24324324324324 ucm 1.89189189189189 ucm 3.51351351351351 ucm 2.16216216216216 ucm box fill
4.32432432432432 ucm 1.89189189189189 ucm 4.59459459459459 ucm 2.16216216216216 ucm box fill
5.40540540540541 ucm 1.89189189189189 ucm 5.67567567567568 ucm 2.16216216216216 ucm box fill
6.48648648648649 ucm 1.89189189189189 ucm 6.75675675675676 ucm 2.16216216216216 ucm box fill
7.56756756756757 ucm 1.89189189189189 ucm 7.83783783783784 ucm 2.16216216216216 ucm box fill
8.64864864864865 ucm 1.89189189189189 ucm 8.91891891891892 ucm 2.16216216216216 ucm box fill
8.91891891891892 ucm 1.89189189189189 ucm 9.18918918918919 ucm 2.16216216216216 ucm box fill
9.45945945945946 ucm 1.89189189189189 ucm 9.72972972972973 ucm 2.16216216216216 ucm box fill
0 ucm 2.16216216216216 ucm 0.27027027027027 ucm 2.43243243243243 ucm box fill
0.27027027027027 ucm 2.16216216216216 ucm 0.540540540540541 ucm 2.43243243243243 ucm box fill
0.540540540540541 ucm 2.16216216216216 ucm 0.810810810810811 ucm 2.43243243243243 ucm box fill
0.810810810810811 ucm 2.16216216216216 ucm 1.08108108108108 ucm 2.43243243243243 ucm box fill
1.35135135135135 ucm 2.16216216216216 ucm 1.62162162162162 ucm 2.43243243243243 ucm box fill
1.62162162162162 ucm 2.16216216216216 ucm 1.89189189189189 ucm 2.43243243243243 ucm box fill
2.16216216216216 ucm 2.16216216216216 ucm 2.43243243243243 ucm 2.43243243243243 ucm box fill
2.7027027027027 ucm 2.16216216216216 ucm 2.97297297297297 ucm 2.43243243243243 ucm box fill
2.97297297297297 ucm 2.16216216216216 ucm 3.24324324324324 ucm 2.43243243243243 ucm box fill
3.24324324324324 ucm 2.16216216216216 ucm 3.51351351351351 ucm 2.43243243243243 ucm box fill
3.51351351351351 ucm 2.16216216216216 ucm 3.78378378378378 ucm 2.43243243243243 ucm box fill
4.05405405405405 ucm 2.16216216216216 ucm 4.32432432432432 ucm 2.43243243243243 ucm box fill
4.32432432432432 ucm 2.16216216216216 ucm 4.59459459459459 ucm 2.43243243243243 ucm box fill
4.59459459459459 ucm 2.16216216216216 ucm 4.86486486486486 ucm 2.43243243243243 ucm box fill
5.13513513513514 ucm 2.16216216216216 ucm 5.40540540540541 ucm 2.43243243243243 ucm box fill
5.40540540540541 ucm 2.16216216216216 ucm 5.67567567567568 ucm 2.43243243243243 ucm box fill
5.67567567567568 ucm 2.16216216216216 ucm 5.94594594594595 ucm 2.43243243243243 ucm box fill
6.21621621621622 ucm 2.16216216216216 ucm 6.48648648648649 ucm 2.43243243243243 ucm box fill
6.48648648648649 ucm 2.16216216216216 ucm 6.75675675675676 ucm 2.43243243243243 ucm box fill
6.75675675675676 ucm 2.16216216216216 ucm 7.02702702702703 ucm 2.43243243243243 ucm box fill
7.2972972972973 ucm 2.16216216216216 ucm 7.56756756756757 ucm 2.43243243243243 ucm box fill
7.56756756756757 ucm 2.16216216216216 ucm 7.83783783783784 ucm 2.43243243243243 ucm box fill
7.83783783783784 ucm 2.16216216216216 ucm 8.10810810810811 ucm 2.43243243243243 ucm box fill
8.10810810810811 ucm 2.16216216216216 ucm 8.37837837837838 ucm 2.43243243243243 ucm box fill
8.37837837837838 ucm 2.16216216216216 ucm 8.64864864864865 ucm 2.43243243243243 ucm box fill
8.64864864864865 ucm 2.16216216216216 ucm 8.91891891891892 ucm 2.43243243243243 ucm box fill
9.72972972972973 ucm 2.16216216216216 ucm 10 ucm 2.43243243243243 ucm box fill
1.89189189189189 ucm 2.43243243243243 ucm 2.16216216216216 ucm 2.7027027027027 ucm box fill
2.16216216216216 ucm 2.43243243243243 ucm 2.43243243243243 ucm 2.7027027027027 ucm box fill
2.7027027027027 ucm 2.43243243243243 ucm 2.97297297297297 ucm 2.7027027027027 ucm box fill
3.51351351351351 ucm 2.43243243243243 ucm 3.78378378378378 ucm 2.7027027027027 ucm box fill
4.59459459459459 ucm 2.43243243243243 ucm 4.86486486486486 ucm 2.7027027027027 ucm box fill
5.67567567567568 ucm 2.43243243243243 ucm 5.94594594594595 ucm 2.7027027027027 ucm box fill
6.75675675675676 ucm 2.43243243243243 ucm 7.02702702702703 ucm 2.7027027027027 ucm box fill
8.10810810810811 ucm 2.43243243243243 ucm 8.37837837837838 ucm 2.7027027027027 ucm box fill
8.37837837837838 ucm 2.43243243243243 ucm 8.64864864864865 ucm 2.7027027027027 ucm box fill
8.91891891891892 ucm 2.43243243243243 ucm 9.18918918918919 ucm 2.7027027027027 ucm box fill
0 ucm 2.7027027027027 ucm 0.27027027027027 ucm 2.97297297297297 ucm box fill
0.27027027027027 ucm 2.7027027027027 ucm 0.540540540540541 ucm 2.97297297297297 ucm box fill
0.540540540540541 ucm 2.7027027027027 ucm 0.810810810810811 ucm 2.97297297297297 ucm box fill
0.810810810810811 ucm 2.7027027027027 ucm 1.08108108108108 ucm 2.97297297297297 ucm box fill
1.35135135135135 ucm 2.7027027027027 ucm 1.62162162162162 ucm 2.97297297297297 ucm box fill
1.62162162162162 ucm 2.7027027027027 ucm 1.89189189189189 ucm 2.97297297297297 ucm box fill
1.89189189189189 ucm 2.7027027027027 ucm 2.16216216216216 ucm 2.97297297297297 ucm box fill
2.16216216216216 ucm 2.7027027027027 ucm 2.43243243243243 ucm 2.97297297297297 ucm box fill
2.43243243243243 ucm 2.7027027027027 ucm 2.7027027027027 ucm 2.97297297297297 ucm box fill
2.97297297297297 ucm 2.7027027027027 ucm 3.24324324324324 ucm 2.97297297297297 ucm box fill
4.05405405405405 ucm 2.7027027027027 ucm 4.32432432432432 ucm 2.97297297297297 ucm box fill
5.13513513513514 ucm 2.7027027027027 ucm 5.40540540540541 ucm 2.97297297297297 ucm box fill
6.21621621621622 ucm 2.7027027027027 ucm 6.48648648648649 ucm 2.97297297297297 ucm box fill
7.2972972972973 ucm 2.7027027027027 ucm 7.56756756756757 ucm 2.97297297297297 ucm box fill
8.37837837837838 ucm 2.7027027027027 ucm 8.64864864864865 ucm 2.97297297297297 ucm box fill
8.64864864864865 ucm 2.7027027027027 ucm 8.91891891891892 ucm 2.97297297297297 ucm box fill
8.91891891891892 ucm 2.7027027027027 ucm 9.18918918918919 ucm 2.97297297297297 ucm box fill
9.18918918918919 ucm 2.7027027027027 ucm 9.45945945945946 ucm 2.97297297297297 ucm box fill
9.72972972972973 ucm 2.7027027027027 ucm 10 ucm 2.97297297297297 ucm box fill
0.27027027027027 ucm 2.97297297297297 ucm 0.540540540540541 ucm 3.24324324324324 ucm box fill
1.08108108108108 ucm 2.97297297297297 ucm 1.35135135135135 ucm 3.24324324324324 ucm box fill
1.35135135135135 ucm 2.97297297297297 ucm 1.62162162162162 ucm 3.24324324324324 ucm box fill
2.16216216216216 ucm 2.97297297297297 ucm 2.43243243243243 ucm 3.24324324324324 ucm box fill
2.7027027027027 ucm 2.97297297297297 ucm 2.97297297297297 ucm 3.24324324324324 ucm box fill
3.51351351351351 ucm 2.97297297297297 ucm 3.78378378378378 ucm 3.24324324324324 ucm box fill
3.78378378378378 ucm 2.97297297297297 ucm 4.05405405405405 ucm 3.24324324324324 ucm box fill
4.59459459459459 ucm 2.97297297297297 ucm 4.86486486486486 ucm 3.24324324324324 ucm box fill
4.86486486486486 ucm 2.97297297297297 ucm 5.13513513513514 ucm 3.24324324324324 ucm box fill
5.67567567567568 ucm 2.97297297297297 ucm 5.94594594594595 ucm 3.24324324324324 ucm box fill
5.94594594594595 ucm 2.97297297297297 ucm 6.21621621621622 ucm 3.24324324324324 ucm box fill
6.75675675675676 ucm 2.97297297297297 ucm 7.02702702702703 ucm 3.24324324324324 ucm box fill
7.02702702702703 ucm 2.97297297297297 ucm 7.2972972972973 ucm 3.24324324324324 ucm box fill
7.83783783783784 ucm 2.97297297297297 ucm 8.10810810810811 ucm 3.24324324324324 ucm box fill
9.72972972972973 ucm 2.97297297297297 ucm 10 ucm 3.24324324324324 ucm box fill
0.540540540540541 ucm 3.24324324324324 ucm 0.810810810810811 ucm 3.51351351351351 ucm box fill
1.08108108108108 ucm 3.24324324324324 ucm 1.35135135135135 ucm 3.51351351351351 ucm box fill
1.35135135135135 ucm 3.24324324324324 ucm 1.62162162162162 ucm 3.51351351351351 ucm box fill
1.62162162162162 ucm 3.24324324324324 ucm 1.89189189189189 ucm 3.51351351351351 ucm box fill
1.89189189189189 ucm 3.24324324324324 ucm 2.16216216216216 ucm 3.51351351351351 ucm box fill
2.16216216216216 ucm 3.24324324324324 ucm 2.43243243243243 ucm 3.51351351351351 ucm box fill
3.51351351351351 ucm 3.24324324324324 ucm 3.78378378378378 ucm 3.51351351351351 ucm box fill
3.78378378378378 ucm 3.24324324324324 ucm 4.05405405405405 ucm 3.51351351351351 ucm box fill
4.59459459459459 ucm 3.24324324324324 ucm 4.86486486486486 ucm 3.51351351351351 ucm box fill
4.86486486486486 ucm 3.24324324324324 ucm 5.13513513513514 ucm 3.51351351351351 ucm box fill
5.67567567567568 ucm 3.24324324324324 ucm 5.94594594594595 ucm 3.51351351351351 ucm box fill
5.94594594594595 ucm 3.24324324324324 ucm 6.21621621621622 ucm 3.51351351351351 ucm box fill
6.75675675675676 ucm 3.24324324324324 ucm 7.02702702702703 ucm 3.51351351351351 ucm box fill
7.02702702702703 ucm 3.24324324324324 ucm 7.2972972972973 ucm 3.51351351351351 ucm box fill
7.83783783783784 ucm 3.24324324324324 ucm 8.10810810810811 ucm 3.51351351351351 ucm box fill
8.10810810810811 ucm 3.24324324324324 ucm 8.37837837837838 ucm 3.51351351351351 ucm box fill
8.37837837837838 ucm 3.24324324324324 ucm 8.64864864864865 ucm 3.51351351351351 ucm box fill
8.64864864864865 ucm 3.24324324324324 ucm 8.91891891891892 ucm 3.51351351351351 ucm box fill
8.91891891891892 ucm 3.24324324324324 ucm 9.18918918918919 ucm 3.51351351351351 ucm box fill
9.45945945945946 ucm 3.24324324324324 ucm 9.72972972972973 ucm 3.51351351351351 ucm box fill
0 ucm 3.51351351351351 ucm 0.27027027027027 ucm 3.78378378378378 ucm box fill
0.27027027027027 ucm 3.51351351351351 ucm 0.540540540540541 ucm 3.78378378378378 ucm box fill
0.540540540540541 ucm 3.51351351351351 ucm 0.810810810810811 ucm 3.78378378378378 ucm box fill
0.810810810810811 ucm 3.51351351351351 ucm 1.08108108108108 ucm 3.78378378378378 ucm box fill
1.35135135135135 ucm 3.51351351351351 ucm 1.62162162162162 ucm 3.78378378378378 ucm box fill
2.16216216216216 ucm 3.51351351351351 ucm 2.43243243243243 ucm 3.78378378378378 ucm box fill
2.7027027027027 ucm 3.51351351351351 ucm 2.97297297297297 ucm 3.78378378378378 ucm box fill
2.97297297297297 ucm 3.51351351351351 ucm 3.24324324324324 ucm 3.78378378378378 ucm box fill
3.24324324324324 ucm 3.51351351351351 ucm 3.51351351351351 ucm 3.78378378378378 ucm box fill
3.78378378378378 ucm 3.51351351351351 ucm 4.05405405405405 ucm 3.78378378378378 ucm box fill
4.05405405405405 ucm 3.51351351351351 ucm 4.32432432432432 ucm 3.78378378378378 ucm box fill
4.32432432432432 ucm 3.51351351351351 ucm 4.59459459459459 ucm 3.78378378378378 ucm box fill
4.86486486486486 ucm 3.51351351351351 ucm 5.13513513513514 ucm 3.78378378378378 ucm box fill
5.13513513513514 ucm 3.51351351351351 ucm 5.40540540540541 ucm 3.78378378378378 ucm box fill
5.40540540540541 ucm 3.51351351351351 ucm 5.67567567567568 ucm 3.78378378378378 ucm box fill
5.94594594594595 ucm 3.51351351351351 ucm 6.21621621621622 ucm 3.78378378378378 ucm box fill
6.21621621621622 ucm 3.51351351351351 ucm 6.48648648648649 ucm 3.78378378378378 ucm box fill
6.48648648648649 ucm 3.51351351351351 ucm 6.75675675675676 ucm 3.78378378378378 ucm box fill
7.02702702702703 ucm 3.51351351351351 ucm 7.2972972972973 ucm 3.78378378378378 ucm box fill
7.2972972972973 ucm 3.51351351351351 ucm 7.56756756756757 ucm 3.78378378378378 ucm box fill
7.56756756756757 ucm 3.51351351351351 ucm 7.83783783783784 ucm 3.78378378378378 ucm box fill
7.83783783783784 ucm 3.51351351351351 ucm 8.10810810810811 ucm 3.78378378378378 ucm box fill
8.37837837837838 ucm 3.51351351351351 ucm 8.64864864864865 ucm 3.78378378378378 ucm box fill
8.64864864864865 ucm 3.51351351351351 ucm 8.91891891891892 ucm 3.78378378378378 ucm box fill
0 ucm 3.78378378378378 ucm 0.27027027027027 ucm 4.05405405405405 ucm box fill
1.08108108108108 ucm 3.78378378378378 ucm 1.35135135135135 ucm 4.05405405405405 ucm box fill
1.62162162162162 ucm 3.78378378378378 ucm 1.89189189189189 ucm 4.05405405405405 ucm box fill
2.16216216216216 ucm 3.78378378378378 ucm 2.43243243243243 ucm 4.05405405405405 ucm box fill
2.43243243243243 ucm 3.78378378378378 ucm 2.7027027027027 ucm 4.05405405405405 ucm box fill
3.24324324324324 ucm 3.78378378378378 ucm 3.51351351351351 ucm 4.05405405405405 ucm box fill
3.51351351351351 ucm 3.78378378378378 ucm 3.78378378378378 ucm 4.05405405405405 ucm box fill
3.78378378378378 ucm 3.78378378378378 ucm 4.05405405405405 ucm 4.05405405405405 ucm box fill
4.32432432432432 ucm 3.78378378378378 ucm 4.59459459459459 ucm 4.05405405405405 ucm box fill
4.59459459459459 ucm 3.78378378378378 ucm 4.86486486486486 ucm 4.05405405405405 ucm box fill
4.86486486486486 ucm 3.78378378378378 ucm 5.13513513513514 ucm 4.05405405405405 ucm box fill
5.40540540540541 ucm 3.78378378378378 ucm 5.67567567567568 ucm 4.05405405405405 ucm box fill
5.67567567567568 ucm 3.78378378378378 ucm 5.94594594594595 ucm 4.05405405405405 ucm box fill
5.94594594594595 ucm 3.78378378378378 ucm 6.21621621621622 ucm 4.05405405405405 ucm box fill
6.48648648648649 ucm 3.78378378378378 ucm 6.75675675675676 ucm 4.05405405405405 ucm box fill
6.75675675675676 ucm 3.78378378378378 ucm 7.02702702702703 ucm 4.05405405405405 ucm box fill
7.02702702702703 ucm 3.78378378378378 ucm 7.2972972972973 ucm 4.05405405405405 ucm box fill
7.56756756756757 ucm 3.78378378378378 ucm 7.83783783783784 ucm 4.05405405405405 ucm box fill
7.83783783783784 ucm 3.78378378378378 ucm 8.10810810810811 ucm 4.05405405405405 ucm box fill
8.10810810810811 ucm 3.78378378378378 ucm 8.37837837837838 ucm 4.05405405405405 ucm box fill
8.37837837837838 ucm 3.78378378378378 ucm 8.64864864864865 ucm 4.05405405405405 ucm box fill
9.18918918918919 ucm 3.78378378378378 ucm 9.45945945945946 ucm 4.05405405405405 ucm box fill
9.72972972972973 ucm 3.78378378378378 ucm 10 ucm 4.05405405405405 ucm box fill
0 ucm 4.05405405405405 ucm 0.27027027027027 ucm 4.32432432432432 ucm box fill
0.27027027027027 ucm 4.05405405405405 ucm 0.540540540540541 ucm 4.32432432432432 ucm box fill
1.89189189189189 ucm 4.05405405405405 ucm 2.16216216216216 ucm 4.32432432432432 ucm box fill
2.16216216216216 ucm 4.05405405405405 ucm 2.43243243243243 ucm 4.32432432432432 ucm box fill
3.24324324324324 ucm 4.05405405405405 ucm 3.51351351351351 ucm 4.32432432432432 ucm box fill
4.32432432432432 ucm 4.05405405405405 ucm 4.59459459459459 ucm 4.32432432432432 ucm box fill
5.40540540540541 ucm 4.05405405405405 ucm 5.67567567567568 ucm 4.32432432432432 ucm box fill
6.48648648648649 ucm 4.05405405405405 ucm 6.75675675675676 ucm 4.32432432432432 ucm box fill
7.56756756756757 ucm 4.05405405405405 ucm 7.83783783783784 ucm 4.32432432432432 ucm box fill
8.37837837837838 ucm 4.05405405405405 ucm 8.64864864864865 ucm 4.32432432432432 ucm box fill
8.64864864864865 ucm 4.05405405405405 ucm 8.91891891891892 ucm 4.32432432432432 ucm box fill
9.72972972972973 ucm 4.05405405405405 ucm 10 ucm 4.32432432432432 ucm box fill
0 ucm 4.32432432432432 ucm 0.27027027027027 ucm 4.59459459459459 ucm box fill
0.540540540540541 ucm 4.32432432432432 ucm 0.810810810810811 ucm 4.59459459459459 ucm box fill
0.810810810810811 ucm 4.32432432432432 ucm 1.08108108108108 ucm 4.59459459459459 ucm box fill
1.62162162162162 ucm 4.32432432432432 ucm 1.89189189189189 ucm 4.59459459459459 ucm box fill
1.89189189189189 ucm 4.32432432432432 ucm 2.16216216216216 ucm 4.59459459459459 ucm box fill
2.97297297297297 ucm 4.32432432432432 ucm 3.24324324324324 ucm 4.59459459459459 ucm box fill
3.24324324324324 ucm 4.32432432432432 ucm 3.51351351351351 ucm 4.59459459459459 ucm box fill
3.51351351351351 ucm 4.32432432432432 ucm 3.78378378378378 ucm 4.59459459459459 ucm box fill
4.05405405405405 ucm 4.32432432432432 ucm 4.32432432432432 ucm 4.59459459459459 ucm box fill
4.32432432432432 ucm 4.32432432432432 ucm 4.59459459459459 ucm 4.59459459459459 ucm box fill
4.59459459459459 ucm 4.32432432432432 ucm 4.86486486486486 ucm 4.59459459459459 ucm box fill
5.13513513513514 ucm 4.32432432432432 ucm 5.40540540540541 ucm 4.59459459459459 ucm box fill
5.40540540540541 ucm 4.32432432432432 ucm 5.67567567567568 ucm 4.59459459459459 ucm box fill
5.67567567567568 ucm 4.32432432432432 ucm 5.94594594594595 ucm 4.59459459459459 ucm box fill
6.21621621621622 ucm 4.32432432432432 ucm 6.48648648648649 ucm 4.59459459459459 ucm box fill
6.48648648648649 ucm 4.32432432432432 ucm 6.75675675675676 ucm 4.59459459459459 ucm box fill
6.75675675675676 ucm 4.32432432432432 ucm 7.02702702702703 ucm 4.59459459459459 ucm box fill
7.2972972972973 ucm 4.32432432432432 ucm 7.56756756756757 ucm 4.59459459459459 ucm box fill
7.56756756756757 ucm 4.32432432432432 ucm 7.83783783783784 ucm 4.59459459459459 ucm box fill
8.37837837837838 ucm 4.32432432432432 ucm 8.64864864864865 ucm 4.59459459459459 ucm box fill
8.64864864864865 ucm 4.32432432432432 ucm 8.91891891891892 ucm 4.59459459459459 ucm box fill
8.91891891891892 ucm 4.32432432432432 ucm 9.18918918918919 ucm 4.59459459459459 ucm box fill
9.45945945945946 ucm 4.32432432432432 ucm 9.72972972972973 ucm 4.59459459459459 ucm box fill
0.810810810810811 ucm 4.59459459459459 ucm 1.08108108108108 ucm 4.86486486486486 ucm box fill
1.08108108108108 ucm 4.59459459459459 ucm 1.35135135135135 ucm 4.86486486486486 ucm box fill
2.16216216216216 ucm 4.59459459459459 ucm 2.43243243243243 ucm 4.86486486486486 ucm box fill
2.43243243243243 ucm 4.59459459459459 ucm 2.7027027027027 ucm 4.86486486486486 ucm box fill
3.51351351351351 ucm 4.59459459459459 ucm 3.78378378378378 ucm 4.86486486486486 ucm box fill
4.59459459459459 ucm 4.59459459459459 ucm 4.86486486486486 ucm 4.86486486486486 ucm box fill
5.67567567567568 ucm 4.59459459459459 ucm 5.94594594594595 ucm 4.86486486486486 ucm box fill
6.75675675675676 ucm 4.59459459459459 ucm 7.02702702702703 ucm 4.86486486486486 ucm box fill
8.10810810810811 ucm 4.59459459459459 ucm 8.37837837837838 ucm 4.86486486486486 ucm box fill
8.91891891891892 ucm 4.59459459459459 ucm 9.18918918918919 ucm 4.86486486486486 ucm box fill
9.45945945945946 ucm 4.59459459459459 ucm 9.72972972972973 ucm 4.86486486486486 ucm box fill
1.08108108108108 ucm 4.86486486486486 ucm 1.35135135135135 ucm 5.13513513513514 ucm box fill
1.35135135135135 ucm 4.86486486486486 ucm 1.62162162162162 ucm 5.13513513513514 ucm box fill
1.62162162162162 ucm 4.86486486486486 ucm 1.89189189189189 ucm 5.13513513513514 ucm box fill
1.89189189189189 ucm 4.86486486486486 ucm 2.16216216216216 ucm 5.13513513513514 ucm box fill
2.16216216216216 ucm 4.86486486486486 ucm 2.43243243243243 ucm 5.13513513513514 ucm box fill
2.43243243243243 ucm 4.86486486486486 ucm 2.7027027027027 ucm 5.13513513513514 ucm box fill
2.97297297297297 ucm 4.86486486486486 ucm 3.24324324324324 ucm 5.13513513513514 ucm box fill
4.05405405405405 ucm 4.86486486486486 ucm 4.32432432432432 ucm 5.13513513513514 ucm box fill
5.13513513513514 ucm 4.86486486486486 ucm 5.40540540540541 ucm 5.13513513513514 ucm box fill
6.21621621621622 ucm 4.86486486486486 ucm 6.48648648648649 ucm 5.13513513513514 ucm box fill
7.2972972972973 ucm 4.86486486486486 ucm 7.56756756756757 ucm 5.13513513513514 ucm box fill
8.64864864864865 ucm 4.86486486486486 ucm 8.91891891891892 ucm 5.13513513513514 ucm box fill
8.91891891891892 ucm 4.86486486486486 ucm 9.18918918918919 ucm 5.13513513513514 ucm box fill
9.18918918918919 ucm 4.86486486486486 ucm 9.45945945945946 ucm 5.13513513513514 ucm box fill
9.72972972972973 ucm 4.86486486486486 ucm 10 ucm 5.13513513513514 ucm box fill
0 ucm 5.13513513513514 ucm 0.27027027027027 ucm 5.40540540540541 ucm box fill
0.27027027027027 ucm 5.13513513513514 ucm 0.540540540540541 ucm 5.40540540540541 ucm box fill
1.08108108108108 ucm 5.13513513513514 ucm 1.35135135135135 ucm 5.40540540540541 ucm box fill
2.16216216216216 ucm 5.13513513513514 ucm 2.43243243243243 ucm 5.40540540540541 ucm box fill
3.51351351351351 ucm 5.13513513513514 ucm 3.78378378378378 ucm 5.40540540540541 ucm box fill
3.78378378378378 ucm 5.13513513513514 ucm 4.05405405405405 ucm 5.40540540540541 ucm box fill
4.59459459459459 ucm 5.13513513513514 ucm 4.86486486486486 ucm 5.40540540540541 ucm box fill
4.86486486486486 ucm 5.13513513513514 ucm 5.13513513513514 ucm 5.40540540540541 ucm box fill
5.67567567567568 ucm 5.13513513513514 ucm 5.94594594594595 ucm 5.40540540540541 ucm box fill
5.94594594594595 ucm 5.13513513513514 ucm 6.21621621621622 ucm 5.40540540540541 ucm box fill
6.75675675675676 ucm 5.13513513513514 ucm 7.02702702702703 ucm 5.40540540540541 ucm box fill
7.02702702702703 ucm 5.13513513513514 ucm 7.2972972972973 ucm 5.40540540540541 ucm box fill
7.83783783783784 ucm 5.13513513513514 ucm 8.10810810810811 ucm 5.40540540540541 ucm box fill
9.45945945945946 ucm 5.13513513513514 ucm 9.72972972972973 ucm 5.40540540540541 ucm box fill
9.72972972972973 ucm 5.13513513513514 ucm 10 ucm 5.40540540540541 ucm box fill
0 ucm 5.40540540540541 ucm 0.27027027027027 ucm 5.67567567567568 ucm box fill
0.27027027027027 ucm 5.40540540540541 ucm 0.540540540540541 ucm 5.67567567567568 ucm box fill
0.540540540540541 ucm 5.40540540540541 ucm 0.810810810810811 ucm 5.67567567567568 ucm box fill
0.810810810810811 ucm 5.40540540540541 ucm 1.08108108108108 ucm 5.67567567567568 ucm box fill
1.08108108108108 ucm 5.40540540540541 ucm 1.35135135135135 ucm 5.67567567567568 ucm box fill
1.62162162162162 ucm 5.40540540540541 ucm 1.89189189189189 ucm 5.67567567567568 ucm box fill
1.89189189189189 ucm 5.40540540540541 ucm 2.16216216216216 ucm 5.67567567567568 ucm box fill
2.43243243243243 ucm 5.40540540540541 ucm 2.7027027027027 ucm 5.67567567567568 ucm box fill
2.7027027027027 ucm 5.40540540540541 ucm 2.97297297297297 ucm 5.67567567567568 ucm box fill
3.51351351351351 ucm 5.40540540540541 ucm 3.78378378378378 ucm 5.67567567567568 ucm box fill
3.78378378378378 ucm 5.40540540540541 ucm 4.05405405405405 ucm 5.67567567567568 ucm box fill
4.59459459459459 ucm 5.40540540540541 ucm 4.86486486486486 ucm 5.67567567567568 ucm box fill
4.86486486486486 ucm 5.40540540540541 ucm 5.13513513513514 ucm 5.67567567567568 ucm box fill
5.67567567567568 ucm 5.40540540540541 ucm 5.94594594594595 ucm 5.67567567567568 ucm box fill
5.94594594594595 ucm 5.40540540540541 ucm 6.21621621621622 ucm 5.67567567567568 ucm box fill
6.75675675675676 ucm 5.40540540540541 ucm 7.02702702702703 ucm 5.67567567567568 ucm box fill
7.02702702702703 ucm 5.40540540540541 ucm 7.2972972972973 ucm 5.67567567567568 ucm box fill
7.83783783783784 ucm 5.40540540540541 ucm 8.10810810810811 ucm 5.67567567567568 ucm box fill
8.10810810810811 ucm 5.40540540540541 ucm 8.37837837837838 ucm 5.67567567567568 ucm box fill
8.64864864864865 ucm 5.40540540540541 ucm 8.91891891891892 ucm 5.67567567567568 ucm box fill
8.91891891891892 ucm 5.40540540540541 ucm 9.18918918918919 ucm 5.67567567567568 ucm box fill
9.45945945945946 ucm 5.40540540540541 ucm 9.72972972972973 ucm 5.67567567567568 ucm box fill
9.72972972972973 ucm 5.40540540540541 ucm 10 ucm 5.67567567567568 ucm box fill
0.27027027027027 ucm 5.67567567567568 ucm 0.540540540540541 ucm 5.94594594594595 ucm box fill
0.540540540540541 ucm 5.67567567567568 ucm 0.810810810810811 ucm 5.94594594594595 ucm box fill
0.810810810810811 ucm 5.67567567567568 ucm 1.08108108108108 ucm 5.94594594594595 ucm box fill
1.08108108108108 ucm 5.67567567567568 ucm 1.35135135135135 ucm 5.94594594594595 ucm box fill
1.35135135135135 ucm 5.67567567567568 ucm 1.62162162162162 ucm 5.94594594594595 ucm box fill
1.89189189189189 ucm 5.67567567567568 ucm 2.16216216216216 ucm 5.94594594594595 ucm box fill
2.16216216216216 ucm 5.67567567567568 ucm 2.43243243243243 ucm 5.94594594594595 ucm box fill
2.7027027027027 ucm 5.67567567567568 ucm 2.97297297297297 ucm 5.94594594594595 ucm box fill
2.97297297297297 ucm 5.67567567567568 ucm 3.24324324324324 ucm 5.94594594594595 ucm box fill
3.24324324324324 ucm 5.67567567567568 ucm 3.51351351351351 ucm 5.94594594594595 ucm box fill
3.78378378378378 ucm 5.67567567567568 ucm 4.05405405405405 ucm 5.94594594594595 ucm box fill
4.05405405405405 ucm 5.67567567567568 ucm 4.32432432432432 ucm 5.94594594594595 ucm box fill
4.32432432432432 ucm 5.67567567567568 ucm 4.59459459459459 ucm 5.94594594594595 ucm box fill
4.86486486486486 ucm 5.67567567567568 ucm 5.13513513513514 ucm 5.94594594594595 ucm box fill
5.13513513513514 ucm 5.67567567567568 ucm 5.40540540540541 ucm 5.94594594594595 ucm box fill
5.40540540540541 ucm 5.67567567567568 ucm 5.67567567567568 ucm 5.94594594594595 ucm box fill
5.94594594594595 ucm 5.67567567567568 ucm 6.21621621621622 ucm 5.94594594594595 ucm box fill
6.21621621621622 ucm 5.67567567567568 ucm 6.48648648648649 ucm 5.94594594594595 ucm box fill
6.48648648648649 ucm 5.67567567567568 ucm 6.75675675675676 ucm 5.94594594594595 ucm box fill
7.02702702702703 ucm 5.67567567567568 ucm 7.2972972972973 ucm 5.94594594594595 ucm box fill
7.2972972972973 ucm 5.67567567567568 ucm 7.56756756756757 ucm 5.94594594594595 ucm box fill
7.56756756756757 ucm 5.67567567567568 ucm 7.83783783783784 ucm 5.94594594594595 ucm box fill
7.83783783783784 ucm 5.67567567567568 ucm 8.10810810810811 ucm 5.94594594594595 ucm box fill
8.37837837837838 ucm 5.67567567567568 ucm 8.64864864864865 ucm 5.94594594594595 ucm box fill
8.64864864864865 ucm 5.67567567567568 ucm 8.91891891891892 ucm 5.94594594594595 ucm box fill
9.45945945945946 ucm 5.67567567567568 ucm 9.72972972972973 ucm 5.94594594594595 ucm box fill
9.72972972972973 ucm 5.67567567567568 ucm 10 ucm 5.94594594594595 ucm box fill
0 ucm 5.94594594594595 ucm 0.27027027027027 ucm 6.21621621621622 ucm box fill
0.27027027027027 ucm 5.94594594594595 ucm 0.540540540540541 ucm 6.21621621621622 ucm box fill
1.08108108108108 ucm 5.94594594594595 ucm 1.35135135135135 ucm 6.21621621621622 ucm box fill
1.35135135135135 ucm 5.94594594594595 ucm 1.62162162162162 ucm 6.21621621621622 ucm box fill
1.62162162162162 ucm 5.94594594594595 ucm 1.89189189189189 ucm 6.21621621621622 ucm box fill
1.89189189189189 ucm 5.94594594594595 ucm 2.16216216216216 ucm 6.21621621621622 ucm box fill
2.7027027027027 ucm 5.94594594594595 ucm 2.97297297297297 ucm 6.21621621621622 ucm box fill
3.24324324324324 ucm 5.94594594594595 ucm 3.51351351351351 ucm 6.21621621621622 ucm box fill
3.51351351351351 ucm 5.94594594594595 ucm 3.78378378378378 ucm 6.21621621621622 ucm box fill
3.78378378378378 ucm 5.94594594594595 ucm 4.05405405405405 ucm 6.21621621621622 ucm box fill
4.32432432432432 ucm 5.94594594594595 ucm 4.59459459459459 ucm 6.21621621621622 ucm box fill
4.59459459459459 ucm 5.94594594594595 ucm 4.86486486486486 ucm 6.21621621621622 ucm box fill
4.86486486486486 ucm 5.94594594594595 ucm 5.13513513513514 ucm 6.21621621621622 ucm box fill
5.40540540540541 ucm 5.94594594594595 ucm 5.67567567567568 ucm 6.21621621621622 ucm box fill
5.67567567567568 ucm 5.94594594594595 ucm 5.94594594594595 ucm 6.21621621621622 ucm box fill
5.94594594594595 ucm 5.94594594594595 ucm 6.21621621621622 ucm 6.21621621621622 ucm box fill
6.48648648648649 ucm 5.94594594594595 ucm 6.75675675675676 ucm 6.21621621621622 ucm box fill
6.75675675675676 ucm 5.94594594594595 ucm 7.02702702702703 ucm 6.21621621621622 ucm box fill
7.02702702702703 ucm 5.94594594594595 ucm 7.2972972972973 ucm 6.21621621621622 ucm box fill
7.56756756756757 ucm 5.94594594594595 ucm 7.83783783783784 ucm 6.21621621621622 ucm box fill
7.83783783783784 ucm 5.94594594594595 ucm 8.10810810810811 ucm 6.21621621621622 ucm box fill
8.10810810810811 ucm 5.94594594594595 ucm 8.37837837837838 ucm 6.21621621621622 ucm box fill
8.37837837837838 ucm 5.94594594594595 ucm 8.64864864864865 ucm 6.21621621621622 ucm box fill
9.45945945945946 ucm 5.94594594594595 ucm 9.72972972972973 ucm 6.21621621621622 ucm box fill
9.72972972972973 ucm 5.94594594594595 ucm 10 ucm 6.21621621621622 ucm box fill
0 ucm 6.21621621621622 ucm 0.27027027027027 ucm 6.48648648648649 ucm box fill
0.540540540540541 ucm 6.21621621621622 ucm 0.810810810810811 ucm 6.48648648648649 ucm box fill
1.35135135135135 ucm 6.21621621621622 ucm 1.62162162162162 ucm 6.48648648648649 ucm box fill
2.7027027027027 ucm 6.21621621621622 ucm 2.97297297297297 ucm 6.48648648648649 ucm box fill
3.24324324324324 ucm 6.21621621621622 ucm 3.51351351351351 ucm 6.48648648648649 ucm box fill
4.32432432432432 ucm 6.21621621621622 ucm 4.59459459459459 ucm 6.48648648648649 ucm box fill
5.40540540540541 ucm 6.21621621621622 ucm 5.67567567567568 ucm 6.48648648648649 ucm box fill
6.48648648648649 ucm 6.21621621621622 ucm 6.75675675675676 ucm 6.48648648648649 ucm box fill
7.56756756756757 ucm 6.21621621621622 ucm 7.83783783783784 ucm 6.48648648648649 ucm box fill
8.37837837837838 ucm 6.21621621621622 ucm 8.64864864864865 ucm 6.48648648648649 ucm box fill
8.64864864864865 ucm 6.21621621621622 ucm 8.91891891891892 ucm 6.48648648648649 ucm box fill
9.72972972972973 ucm 6.21621621621622 ucm 10 ucm 6.48648648648649 ucm box fill
0 ucm 6.48648648648649 ucm 0.27027027027027 ucm 6.75675675675676 ucm box fill
0.27027027027027 ucm 6.48648648648649 ucm 0.540540540540541 ucm 6.75675675675676 ucm box fill
0.540540540540541 ucm 6.48648648648649 ucm 0.810810810810811 ucm 6.75675675675676 ucm box fill
1.35135135135135 ucm 6.48648648648649 ucm 1.62162162162162 ucm 6.75675675675676 ucm box fill
1.62162162162162 ucm 6.48648648648649 ucm 1.89189189189189 ucm 6.75675675675676 ucm box fill
1.89189189189189 ucm 6.48648648648649 ucm 2.16216216216216 ucm 6.75675675675676 ucm box fill
2.97297297297297 ucm 6.48648648648649 ucm 3.24324324324324 ucm 6.75675675675676 ucm box fill
3.24324324324324 ucm 6.48648648648649 ucm 3.51351351351351 ucm 6.75675675675676 ucm box fill
3.51351351351351 ucm 6.48648648648649 ucm 3.78378378378378 ucm 6.75675675675676 ucm box fill
4.05405405405405 ucm 6.48648648648649 ucm 4.32432432432432 ucm 6.75675675675676 ucm box fill
4.32432432432432 ucm 6.48648648648649 ucm 4.59459459459459 ucm 6.75675675675676 ucm box fill
4.59459459459459 ucm 6.48648648648649 ucm 4.86486486486486 ucm 6.75675675675676 ucm box fill
5.13513513513514 ucm 6.48648648648649 ucm 5.40540540540541 ucm 6.75675675675676 ucm box fill
5.40540540540541 ucm 6.48648648648649 ucm 5.67567567567568 ucm 6.75675675675676 ucm box fill
5.67567567567568 ucm 6.48648648648649 ucm 5.94594594594595 ucm 6.75675675675676 ucm box fill
6.21621621621622 ucm 6.48648648648649 ucm 6.48648648648649 ucm 6.75675675675676 ucm box fill
6.48648648648649 ucm 6.48648648648649 ucm 6.75675675675676 ucm 6.75675675675676 ucm box fill
6.75675675675676 ucm 6.48648648648649 ucm 7.02702702702703 ucm 6.75675675675676 ucm box fill
7.2972972972973 ucm 6.48648648648649 ucm 7.56756756756757 ucm 6.75675675675676 ucm box fill
7.56756756756757 ucm 6.48648648648649 ucm 7.83783783783784 ucm 6.75675675675676 ucm box fill
8.37837837837838 ucm 6.48648648648649 ucm 8.64864864864865 ucm 6.75675675675676 ucm box fill
8.64864864864865 ucm 6.48648648648649 ucm 8.91891891891892 ucm 6.75675675675676 ucm box fill
9.45945945945946 ucm 6.48648648648649 ucm 9.72972972972973 ucm 6.75675675675676 ucm box fill
9.72972972972973 ucm 6.48648648648649 ucm 10 ucm 6.75675675675676 ucm box fill
0.27027027027027 ucm 6.75675675675676 ucm 0.540540540540541 ucm 7.02702702702703 ucm box fill
0.540540540540541 ucm 6.75675675675676 ucm 0.810810810810811 ucm 7.02702702702703 ucm box fill
0.810810810810811 ucm 6.75675675675676 ucm 1.08108108108108 ucm 7.02702702702703 ucm box fill
1.08108108108108 ucm 6.75675675675676 ucm 1.35135135135135 ucm 7.02702702702703 ucm box fill
1.89189189189189 ucm 6.75675675675676 ucm 2.16216216216216 ucm 7.02702702702703 ucm box fill
2.16216216216216 ucm 6.75675675675676 ucm 2.43243243243243 ucm 7.02702702702703 ucm box fill
2.43243243243243 ucm 6.75675675675676 ucm 2.7027027027027 ucm 7.02702702702703 ucm box fill
3.51351351351351 ucm 6.75675675675676 ucm 3.78378378378378 ucm 7.02702702702703 ucm box fill
4.59459459459459 ucm 6.75675675675676 ucm 4.86486486486486 ucm 7.02702702702703 ucm box fill
5.67567567567568 ucm 6.75675675675676 ucm 5.94594594594595 ucm 7.02702702702703 ucm box fill
6.75675675675676 ucm 6.75675675675676 ucm 7.02702702702703 ucm 7.02702702702703 ucm box fill
8.10810810810811 ucm 6.75675675675676 ucm 8.37837837837838 ucm 7.02702702702703 ucm box fill
8.91891891891892 ucm 6.75675675675676 ucm 9.18918918918919 ucm 7.02702702702703 ucm box fill
0.27027027027027 ucm 7.02702702702703 ucm 0.540540540540541 ucm 7.2972972972973 ucm box fill
1.08108108108108 ucm 7.02702702702703 ucm 1.35135135135135 ucm 7.2972972972973 ucm box fill
1.35135135135135 ucm 7.02702702702703 ucm 1.62162162162162 ucm 7.2972972972973 ucm box fill
1.62162162162162 ucm 7.02702702702703 ucm 1.89189189189189 ucm 7.2972972972973 ucm box fill
2.16216216216216 ucm 7.02702702702703 ucm 2.43243243243243 ucm 7.2972972972973 ucm box fill
2.7027027027027 ucm 7.02702702702703 ucm 2.97297297297297 ucm 7.2972972972973 ucm box fill
2.97297297297297 ucm 7.02702702702703 ucm 3.24324324324324 ucm 7.2972972972973 ucm box fill
4.05405405405405 ucm 7.02702702702703 ucm 4.32432432432432 ucm 7.2972972972973 ucm box fill
5.13513513513514 ucm 7.02702702702703 ucm 5.40540540540541 ucm 7.2972972972973 ucm box fill
6.21621621621622 ucm 7.02702702702703 ucm 6.48648648648649 ucm 7.2972972972973 ucm box fill
7.2972972972973 ucm 7.02702702702703 ucm 7.56756756756757 ucm 7.2972972972973 ucm box fill
8.64864864864865 ucm 7.02702702702703 ucm 8.91891891891892 ucm 7.2972972972973 ucm box fill
9.18918918918919 ucm 7.02702702702703 ucm 9.45945945945946 ucm 7.2972972972973 ucm box fill
9.72972972972973 ucm 7.02702702702703 ucm 10 ucm 7.2972972972973 ucm box fill
0.27027027027027 ucm 7.2972972972973 ucm 0.540540540540541 ucm 7.56756756756757 ucm box fill
1.08108108108108 ucm 7.2972972972973 ucm 1.35135135135135 ucm 7.56756756756757 ucm box fill
1.35135135135135 ucm 7.2972972972973 ucm 1.62162162162162 ucm 7.56756756756757 ucm box fill
2.7027027027027 ucm 7.2972972972973 ucm 2.97297297297297 ucm 7.56756756756757 ucm box fill
3.51351351351351 ucm 7.2972972972973 ucm 3.78378378378378 ucm 7.56756756756757 ucm box fill
3.78378378378378 ucm 7.2972972972973 ucm 4.05405405405405 ucm 7.56756756756757 ucm box fill
4.59459459459459 ucm 7.2972972972973 ucm 4.86486486486486 ucm 7.56756756756757 ucm box fill
4.86486486486486 ucm 7.2972972972973 ucm 5.13513513513514 ucm 7.56756756756757 ucm box fill
5.67567567567568 ucm 7.2972972972973 ucm 5.94594594594595 ucm 7.56756756756757 ucm box fill
5.94594594594595 ucm 7.2972972972973 ucm 6.21621621621622 ucm 7.56756756756757 ucm box fill
6.75675675675676 ucm 7.2972972972973 ucm 7.02702702702703 ucm 7.56756756756757 ucm box fill
7.02702702702703 ucm 7.2972972972973 ucm 7.2972972972973 ucm 7.56756756756757 ucm box fill
7.83783783783784 ucm 7.2972972972973 ucm 8.10810810810811 ucm 7.56756756756757 ucm box fill
9.18918918918919 ucm 7.2972972972973 ucm 9.45945945945946 ucm 7.56756756756757 ucm box fill
9.72972972972973 ucm 7.2972972972973 ucm 10 ucm 7.56756756756757 ucm box fill
0 ucm 7.56756756756757 ucm 0.27027027027027 ucm 7.83783783783784 ucm box fill
0.27027027027027 ucm 7.56756756756757 ucm 0.540540540540541 ucm 7.83783783783784 ucm box fill
0.540540540540541 ucm 7.56756756756757 ucm 0.810810810810811 ucm 7.83783783783784 ucm box fill
1.35135135135135 ucm 7.56756756756757 ucm 1.62162162162162 ucm 7.83783783783784 ucm box fill
1.62162162162162 ucm 7.56756756756757 ucm 1.89189189189189 ucm 7.83783783783784 ucm box fill
2.16216216216216 ucm 7.56756756756757 ucm 2.43243243243243 ucm 7.83783783783784 ucm box fill
2.43243243243243 ucm 7.56756756756757 ucm 2.7027027027027 ucm 7.83783783783784 ucm box fill
2.7027027027027 ucm 7.56756756756757 ucm 2.97297297297297 ucm 7.83783783783784 ucm box fill
3.51351351351351 ucm 7.56756756756757 ucm 3.78378378378378 ucm 7.83783783783784 ucm box fill
3.78378378378378 ucm 7.56756756756757 ucm 4.05405405405405 ucm 7.83783783783784 ucm box fill
4.59459459459459 ucm 7.56756756756757 ucm 4.86486486486486 ucm 7.83783783783784 ucm box fill
4.86486486486486 ucm 7.56756756756757 ucm 5.13513513513514 ucm 7.83783783783784 ucm box fill
5.67567567567568 ucm 7.56756756756757 ucm 5.94594594594595 ucm 7.83783783783784 ucm box fill
5.94594594594595 ucm 7.56756756756757 ucm 6.21621621621622 ucm 7.83783783783784 ucm box fill
6.75675675675676 ucm 7.56756756756757 ucm 7.02702702702703 ucm 7.83783783783784 ucm box fill
7.02702702702703 ucm 7.56756756756757 ucm 7.2972972972973 ucm 7.83783783783784 ucm box fill
7.83783783783784 ucm 7.56756756756757 ucm 8.10810810810811 ucm 7.83783783783784 ucm box fill
8.10810810810811 ucm 7.56756756756757 ucm 8.37837837837838 ucm 7.83783783783784 ucm box fill
8.37837837837838 ucm 7.56756756756757 ucm 8.64864864864865 ucm 7.83783783783784 ucm box fill
8.64864864864865 ucm 7.56756756756757 ucm 8.91891891891892 ucm 7.83783783783784 ucm box fill
9.45945945945946 ucm 7.56756756756757 ucm 9.72972972972973 ucm 7.83783783783784 ucm box fill
9.72972972972973 ucm 7.56756756756757 ucm 10 ucm 7.83783783783784 ucm box fill
2.16216216216216 ucm 7.83783783783784 ucm 2.43243243243243 ucm 8.10810810810811 ucm box fill
2.7027027027027 ucm 7.83783783783784 ucm 2.97297297297297 ucm 8.10810810810811 ucm box fill
2.97297297297297 ucm 7.83783783783784 ucm 3.24324324324324 ucm 8.10810810810811 ucm box fill
3.24324324324324 ucm 7.83783783783784 ucm 3.51351351351351 ucm 8.10810810810811 ucm box fill
3.78378378378378 ucm 7.83783783783784 ucm 4.05405405405405 ucm 8.10810810810811 ucm box fill
4.05405405405405 ucm 7.83783783783784 ucm 4.32432432432432 ucm 8.10810810810811 ucm box fill
4.32432432432432 ucm 7.83783783783784 ucm 4.59459459459459 ucm 8.10810810810811 ucm box fill
4.86486486486486 ucm 7.83783783783784 ucm 5.13513513513514 ucm 8.10810810810811 ucm box fill
5.13513513513514 ucm 7.83783783783784 ucm 5.40540540540541 ucm 8.10810810810811 ucm box fill
5.40540540540541 ucm 7.83783783783784 ucm 5.67567567567568 ucm 8.10810810810811 ucm box fill
5.94594594594595 ucm 7.83783783783784 ucm 6.21621621621622 ucm 8.10810810810811 ucm box fill
6.21621621621622 ucm 7.83783783783784 ucm 6.48648648648649 ucm 8.10810810810811 ucm box fill
6.48648648648649 ucm 7.83783783783784 ucm 6.75675675675676 ucm 8.10810810810811 ucm box fill
7.02702702702703 ucm 7.83783783783784 ucm 7.2972972972973 ucm 8.10810810810811 ucm box fill
7.2972972972973 ucm 7.83783783783784 ucm 7.56756756756757 ucm 8.10810810810811 ucm box fill
7.56756756756757 ucm 7.83783783783784 ucm 7.83783783783784 ucm 8.10810810810811 ucm box fill
0 ucm 8.10810810810811 ucm 0.27027027027027 ucm 8.37837837837838 ucm box fill
0.27027027027027 ucm 8.10810810810811 ucm 0.540540540540541 ucm 8.37837837837838 ucm box fill
0.540540540540541 ucm 8.10810810810811 ucm 0.810810810810811 ucm 8.37837837837838 ucm box fill
0.810810810810811 ucm 8.10810810810811 ucm 1.08108108108108 ucm 8.37837837837838 ucm box fill
1.08108108108108 ucm 8.10810810810811 ucm 1.35135135135135 ucm 8.37837837837838 ucm box fill
1.35135135135135 ucm 8.10810810810811 ucm 1.62162162162162 ucm 8.37837837837838 ucm box fill
1.62162162162162 ucm 8.10810810810811 ucm 1.89189189189189 ucm 8.37837837837838 ucm box fill
2.16216216216216 ucm 8.10810810810811 ucm 2.43243243243243 ucm 8.37837837837838 ucm box fill
2.7027027027027 ucm 8.10810810810811 ucm 2.97297297297297 ucm 8.37837837837838 ucm box fill
3.24324324324324 ucm 8.10810810810811 ucm 3.51351351351351 ucm 8.37837837837838 ucm box fill
3.78378378378378 ucm 8.10810810810811 ucm 4.05405405405405 ucm 8.37837837837838 ucm box fill
4.32432432432432 ucm 8.10810810810811 ucm 4.59459459459459 ucm 8.37837837837838 ucm box fill
4.86486486486486 ucm 8.10810810810811 ucm 5.13513513513514 ucm 8.37837837837838 ucm box fill
5.40540540540541 ucm 8.10810810810811 ucm 5.67567567567568 ucm 8.37837837837838 ucm box fill
5.94594594594595 ucm 8.10810810810811 ucm 6.21621621621622 ucm 8.37837837837838 ucm box fill
6.48648648648649 ucm 8.10810810810811 ucm 6.75675675675676 ucm 8.37837837837838 ucm box fill
7.02702702702703 ucm 8.10810810810811 ucm 7.2972972972973 ucm 8.37837837837838 ucm box fill
7.56756756756757 ucm 8.10810810810811 ucm 7.83783783783784 ucm 8.37837837837838 ucm box fill
8.10810810810811 ucm 8.10810810810811 ucm 8.37837837837838 ucm 8.37837837837838 ucm box fill
8.37837837837838 ucm 8.10810810810811 ucm 8.64864864864865 ucm 8.37837837837838 ucm box fill
8.64864864864865 ucm 8.10810810810811 ucm 8.91891891891892 ucm 8.37837837837838 ucm box fill
8.91891891891892 ucm 8.10810810810811 ucm 9.18918918918919 ucm 8.37837837837838 ucm box fill
9.18918918918919 ucm 8.10810810810811 ucm 9.45945945945946 ucm 8.37837837837838 ucm box fill
9.45945945945946 ucm 8.10810810810811 ucm 9.72972972972973 ucm 8.37837837837838 ucm box fill
9.72972972972973 ucm 8.10810810810811 ucm 10 ucm 8.37837837837838 ucm box fill
0 ucm 8.37837837837838 ucm 0.27027027027027 ucm 8.64864864864865 ucm box fill
1.62162162162162 ucm 8.37837837837838 ucm 1.89189189189189 ucm 8.64864864864865 ucm box fill
2.16216216216216 ucm 8.37837837837838 ucm 2.43243243243243 ucm 8.64864864864865 ucm box fill
2.97297297297297 ucm 8.37837837837838 ucm 3.24324324324324 ucm 8.64864864864865 ucm box fill
4.05405405405405 ucm 8.37837837837838 ucm 4.32432432432432 ucm 8.64864864864865 ucm box fill
5.13513513513514 ucm 8.37837837837838 ucm 5.40540540540541 ucm 8.64864864864865 ucm box fill
6.21621621621622 ucm 8.37837837837838 ucm 6.48648648648649 ucm 8.64864864864865 ucm box fill
7.2972972972973 ucm 8.37837837837838 ucm 7.56756756756757 ucm 8.64864864864865 ucm box fill
8.10810810810811 ucm 8.37837837837838 ucm 8.37837837837838 ucm 8.64864864864865 ucm box fill
9.72972972972973 ucm 8.37837837837838 ucm 10 ucm 8.64864864864865 ucm box fill
0 ucm 8.64864864864865 ucm 0.27027027027027 ucm 8.91891891891892 ucm box fill
0.540540540540541 ucm 8.64864864864865 ucm 0.810810810810811 ucm 8.91891891891892 ucm box fill
0.810810810810811 ucm 8.64864864864865 ucm 1.08108108108108 ucm 8.91891891891892 ucm box fill
1.08108108108108 ucm 8.64864864864865 ucm 1.35135135135135 ucm 8.91891891891892 ucm box fill
1.62162162162162 ucm 8.64864864864865 ucm 1.89189189189189 ucm 8.91891891891892 ucm box fill
2.16216216216216 ucm 8.64864864864865 ucm 2.43243243243243 ucm 8.91891891891892 ucm box fill
2.43243243243243 ucm 8.64864864864865 ucm 2.7027027027027 ucm 8.91891891891892 ucm box fill
2.7027027027027 ucm 8.64864864864865 ucm 2.97297297297297 ucm 8.91891891891892 ucm box fill
2.97297297297297 ucm 8.64864864864865 ucm 3.24324324324324 ucm 8.91891891891892 ucm box fill
3.51351351351351 ucm 8.64864864864865 ucm 3.78378378378378 ucm 8.91891891891892 ucm box fill
3.78378378378378 ucm 8.64864864864865 ucm 4.05405405405405 ucm 8.91891891891892 ucm box fill
4.05405405405405 ucm 8.64864864864865 ucm 4.32432432432432 ucm 8.91891891891892 ucm box fill
4.59459459459459 ucm 8.64864864864865 ucm 4.86486486486486 ucm 8.91891891891892 ucm box fill
4.86486486486486 ucm 8.64864864864865 ucm 5.13513513513514 ucm 8.91891891891892 ucm box fill
5.13513513513514 ucm 8.64864864864865 ucm 5.40540540540541 ucm 8.91891891891892 ucm box fill
5.67567567567568 ucm 8.64864864864865 ucm 5.94594594594595 ucm 8.91891891891892 ucm box fill
5.94594594594595 ucm 8.64864864864865 ucm 6.21621621621622 ucm 8.91891891891892 ucm box fill
6.21621621621622 ucm 8.64864864864865 ucm 6.48648648648649 ucm 8.91891891891892 ucm box fill
6.75675675675676 ucm 8.64864864864865 ucm 7.02702702702703 ucm 8.91891891891892 ucm box fill
7.02702702702703 ucm 8.64864864864865 ucm 7.2972972972973 ucm 8.91891891891892 ucm box fill
7.2972972972973 ucm 8.64864864864865 ucm 7.56756756756757 ucm 8.91891891891892 ucm box fill
8.10810810810811 ucm 8.64864864864865 ucm 8.37837837837838 ucm 8.91891891891892 ucm box fill
8.64864864864865 ucm 8.64864864864865 ucm 8.91891891891892 ucm 8.91891891891892 ucm box fill
8.91891891891892 ucm 8.64864864864865 ucm 9.18918918918919 ucm 8.91891891891892 ucm box fill
9.18918918918919 ucm 8.64864864864865 ucm 9.45945945945946 ucm 8.91891891891892 ucm box fill
9.72972972972973 ucm 8.64864864864865 ucm 10 ucm 8.91891891891892 ucm box fill
0 ucm 8.91891891891892 ucm 0.27027027027027 ucm 9.18918918918919 ucm box fill
0.540540540540541 ucm 8.91891891891892 ucm 0.810810810810811 ucm 9.18918918918919 ucm box fill
0.810810810810811 ucm 8.91891891891892 ucm 1.08108108108108 ucm 9.18918918918919 ucm box fill
1.08108108108108 ucm 8.91891891891892 ucm 1.35135135135135 ucm 9.18918918918919 ucm box fill
1.62162162162162 ucm 8.91891891891892 ucm 1.89189189189189 ucm 9.18918918918919 ucm box fill
2.7027027027027 ucm 8.91891891891892 ucm 2.97297297297297 ucm 9.18918918918919 ucm box fill
3.78378378378378 ucm 8.91891891891892 ucm 4.05405405405405 ucm 9.18918918918919 ucm box fill
4.86486486486486 ucm 8.91891891891892 ucm 5.13513513513514 ucm 9.18918918918919 ucm box fill
5.94594594594595 ucm 8.91891891891892 ucm 6.21621621621622 ucm 9.18918918918919 ucm box fill
7.02702702702703 ucm 8.91891891891892 ucm 7.2972972972973 ucm 9.18918918918919 ucm box fill
8.10810810810811 ucm 8.91891891891892 ucm 8.37837837837838 ucm 9.18918918918919 ucm box fill
8.64864864864865 ucm 8.91891891891892 ucm 8.91891891891892 ucm 9.18918918918919 ucm box fill
8.91891891891892 ucm 8.91891891891892 ucm 9.18918918918919 ucm 9.18918918918919 ucm box fill
9.18918918918919 ucm 8.91891891891892 ucm 9.45945945945946 ucm 9.18918918918919 ucm box fill
9.72972972972973 ucm 8.91891891891892 ucm 10 ucm 9.18918918918919 ucm box fill
0 ucm 9.18918918918919 ucm 0.27027027027027 ucm 9.45945945945946 ucm box fill
0.540540540540541 ucm 9.18918918918919 ucm 0.810810810810811 ucm 9.45945945945946 ucm box fill
0.810810810810811 ucm 9.18918918918919 ucm 1.08108108108108 ucm 9.45945945945946 ucm box fill
1.08108108108108 ucm 9.18918918918919 ucm 1.35135135135135 ucm 9.45945945945946 ucm box fill
1.62162162162162 ucm 9.18918918918919 ucm 1.89189189189189 ucm 9.45945945945946 ucm box fill
2.7027027027027 ucm 9.18918918918919 ucm 2.97297297297297 ucm 9.45945945945946 ucm box fill
2.97297297297297 ucm 9.18918918918919 ucm 3.24324324324324 ucm 9.45945945945946 ucm box fill
3.24324324324324 ucm 9.18918918918919 ucm 3.51351351351351 ucm 9.45945945945946 ucm box fill
3.78378378378378 ucm 9.18918918918919 ucm 4.05405405405405 ucm 9.45945945945946 ucm box fill
4.05405405405405 ucm 9.18918918918919 ucm 4.32432432432432 ucm 9.45945945945946 ucm box fill
4.32432432432432 ucm 9.18918918918919 ucm 4.59459459459459 ucm 9.45945945945946 ucm box fill
4.86486486486486 ucm 9.18918918918919 ucm 5.13513513513514 ucm 9.45945945945946 ucm box fill
5.13513513513514 ucm 9.18918918918919 ucm 5.40540540540541 ucm 9.45945945945946 ucm box fill
5.40540540540541 ucm 9.18918918918919 ucm 5.67567567567568 ucm 9.45945945945946 ucm box fill
5.94594594594595 ucm 9.18918918918919 ucm 6.21621621621622 ucm 9.45945945945946 ucm box fill
6.21621621621622 ucm 9.18918918918919 ucm 6.48648648648649 ucm 9.45945945945946 ucm box fill
6.48648648648649 ucm 9.18918918918919 ucm 6.75675675675676 ucm 9.45945945945946 ucm box fill
7.02702702702703 ucm 9.18918918918919 ucm 7.2972972972973 ucm 9.45945945945946 ucm box fill
7.2972972972973 ucm 9.18918918918919 ucm 7.56756756756757 ucm 9.45945945945946 ucm box fill
7.56756756756757 ucm 9.18918918918919 ucm 7.83783783783784 ucm 9.45945945945946 ucm box fill
8.10810810810811 ucm 9.18918918918919 ucm 8.37837837837838 ucm 9.45945945945946 ucm box fill
8.64864864864865 ucm 9.18918918918919 ucm 8.91891891891892 ucm 9.45945945945946 ucm box fill
8.91891891891892 ucm 9.18918918918919 ucm 9.18918918918919 ucm 9.45945945945946 ucm box fill
9.18918918918919 ucm 9.18918918918919 ucm 9.45945945945946 ucm 9.45945945945946 ucm box fill
9.72972972972973 ucm 9.18918918918919 ucm 10 ucm 9.45945945945946 ucm box fill
0 ucm 9.45945945945946 ucm 0.27027027027027 ucm 9.72972972972973 ucm box fill
1.62162162162162 ucm 9.45945945945946 ucm 1.89189189189189 ucm 9.72972972972973 ucm box fill
2.16216216216216 ucm 9.45945945945946 ucm 2.43243243243243 ucm 9.72972972972973 ucm box fill
2.43243243243243 ucm 9.45945945945946 ucm 2.7027027027027 ucm 9.72972972972973 ucm box fill
2.7027027027027 ucm 9.45945945945946 ucm 2.97297297297297 ucm 9.72972972972973 ucm box fill
3.24324324324324 ucm 9.45945945945946 ucm 3.51351351351351 ucm 9.72972972972973 ucm box fill
3.51351351351351 ucm 9.45945945945946 ucm 3.78378378378378 ucm 9.72972972972973 ucm box fill
3.78378378378378 ucm 9.45945945945946 ucm 4.05405405405405 ucm 9.72972972972973 ucm box fill
4.32432432432432 ucm 9.45945945945946 ucm 4.59459459459459 ucm 9.72972972972973 ucm box fill
4.59459459459459 ucm 9.45945945945946 ucm 4.86486486486486 ucm 9.72972972972973 ucm box fill
4.86486486486486 ucm 9.45945945945946 ucm 5.13513513513514 ucm 9.72972972972973 ucm box fill
5.40540540540541 ucm 9.45945945945946 ucm 5.67567567567568 ucm 9.72972972972973 ucm box fill
5.67567567567568 ucm 9.45945945945946 ucm 5.94594594594595 ucm 9.72972972972973 ucm box fill
5.94594594594595 ucm 9.45945945945946 ucm 6.21621621621622 ucm 9.72972972972973 ucm box fill
6.48648648648649 ucm 9.45945945945946 ucm 6.75675675675676 ucm 9.72972972972973 ucm box fill
6.75675675675676 ucm 9.45945945945946 ucm 7.02702702702703 ucm 9.72972972972973 ucm box fill
7.02702702702703 ucm 9.45945945945946 ucm 7.2972972972973 ucm 9.72972972972973 ucm box fill
7.56756756756757 ucm 9.45945945945946 ucm 7.83783783783784 ucm 9.72972972972973 ucm box fill
8.10810810810811 ucm 9.45945945945946 ucm 8.37837837837838 ucm 9.72972972972973 ucm box fill
9.72972972972973 ucm 9.45945945945946 ucm 10 ucm 9.72972972972973 ucm box fill
0 ucm 9.72972972972973 ucm 0.27027027027027 ucm 10 ucm box fill
0.27027027027027 ucm 9.72972972972973 ucm 0.540540540540541 ucm 10 ucm box fill
0.540540540540541 ucm 9.72972972972973 ucm 0.810810810810811 ucm 10 ucm box fill
0.810810810810811 ucm 9.72972972972973 ucm 1.08108108108108 ucm 10 ucm box fill
1.08108108108108 ucm 9.72972972972973 ucm 1.35135135135135 ucm 10 ucm box fill
1.35135135135135 ucm 9.72972972972973 ucm 1.62162162162162 ucm 10 ucm box fill
1.62162162162162 ucm 9.72972972972973 ucm 1.89189189189189 ucm 10 ucm box fill
2.16216216216216 ucm 9.72972972972973 ucm 2.43243243243243 ucm 10 ucm box fill
2.97297297297297 ucm 9.72972972972973 ucm 3.24324324324324 ucm 10 ucm box fill
3.24324324324324 ucm 9.72972972972973 ucm 3.51351351351351 ucm 10 ucm box fill
4.05405405405405 ucm 9.72972972972973 ucm 4.32432432432432 ucm 10 ucm box fill
4.32432432432432 ucm 9.72972972972973 ucm 4.59459459459459 ucm 10 ucm box fill
5.13513513513514 ucm 9.72972972972973 ucm 5.40540540540541 ucm 10 ucm box fill
5.40540540540541 ucm 9.72972972972973 ucm 5.67567567567568 ucm 10 ucm box fill
6.21621621621622 ucm 9.72972972972973 ucm 6.48648648648649 ucm 10 ucm box fill
6.48648648648649 ucm 9.72972972972973 ucm 6.75675675675676 ucm 10 ucm box fill
7.2972972972973 ucm 9.72972972972973 ucm 7.56756756756757 ucm 10 ucm box fill
7.56756756756757 ucm 9.72972972972973 ucm 7.83783783783784 ucm 10 ucm box fill
8.10810810810811 ucm 9.72972972972973 ucm 8.37837837837838 ucm 10 ucm box fill
8.37837837837838 ucm 9.72972972972973 ucm 8.64864864864865 ucm 10 ucm box fill
8.64864864864865 ucm 9.72972972972973 ucm 8.91891891891892 ucm 10 ucm box fill
8.91891891891892 ucm 9.72972972972973 ucm 9.18918918918919 ucm 10 ucm box fill
9.18918918918919 ucm 9.72972972972973 ucm 9.45945945945946 ucm 10 ucm box fill
9.45945945945946 ucm 9.72972972972973 ucm 9.72972972972973 ucm 10 ucm box fill
9.72972972972973 ucm 9.72972972972973 ucm 10 ucm 10 ucm box fill
%%EOF
