import os
from pathlib import Path
from unittest.mock import patch

import pytest
from packaging.version import parse

import setupmeta
import setupmeta.versioning
from setupmeta.model import SetupMeta
from setupmeta.scm import Version

from . import conftest

DEFAULT_BRANCH_SPEC = f"branch({setupmeta.versioning.DEFAULT_BRANCHES})"


def new_meta(versioning, name="just-testing", scm=None, setup_py=None, **kwargs):
    setup_py = setup_py or conftest.resource("setup.py")
    upstream = {"versioning": versioning, "scm": scm, "_setup_py_path": setup_py}
    if name:
        # Allow to test "missing name" case
        upstream["name"] = name

    upstream.update(kwargs)
    return SetupMeta().finalize(upstream=upstream)


def check_strategy(versioning, expected):
    assert str(versioning.strategy) == f"{DEFAULT_BRANCH_SPEC}:{expected}"


def test_deprecated_strategy_notation():
    """Custom separators, and the `!` marker will be removed in the future"""
    with conftest.capture_output() as logged:
        meta = new_meta("post !", scm=conftest.MockGit())
        versioning = meta.versioning
        check_strategy(versioning, "{major}.{minor}.{patch}{post}")
        check_render(versioning, "1.0.0")
        assert "PEP-440 allows only '+' as local" in logged
        assert "'!' character in 'versioning' is now deprecated" in logged

    with conftest.capture_output() as logged:
        meta = new_meta("distance+!foo", scm=conftest.MockGit())
        versioning = meta.versioning
        check_strategy(versioning, "{major}.{minor}.{distance}+foo")
        check_render(versioning, "1.0.0+foo")
        check_render(versioning, "1.0.2+foo", distance=2)
        assert "PEP-440 allows" not in logged
        assert "'!' character in 'versioning' is now deprecated" in logged


def test_disabled(monkeypatch):
    monkeypatch.setattr(setupmeta, "TRACE_ENABLED", True)
    with conftest.capture_output() as out:
        meta = new_meta(False)
        versioning = meta.versioning
        logged = out.pop()
        assert ":: versioning given: 'None', strategy: [None], problem: [setupmeta versioning not enabled]" in logged
        assert not versioning.enabled
        assert versioning.problem == "setupmeta versioning not enabled"
        with pytest.raises(setupmeta.UsageError, match="versioning not enabled"):
            versioning.bump("major", commit=False)


def test_project_scm(sample_project):
    assert setupmeta.versioning.find_scm_root(None, ".git") is None
    assert setupmeta.versioning.find_scm_root("", ".git") is None
    assert setupmeta.versioning.find_scm_root("/", ".git") is None

    assert setupmeta.versioning.find_scm_root(".", ".git") == "."
    assert setupmeta.versioning.find_scm_root("./sub-folder", ".git") == "."

    assert setupmeta.versioning.find_scm_root(sample_project, ".git") == sample_project
    assert setupmeta.versioning.find_scm_root(os.path.join(sample_project, "sub-folder", "foo"), ".git") == sample_project


def test_snapshot_with_version_file():
    with setupmeta.temp_resource() as temp, conftest.capture_output():
        version_file = Path(temp) / setupmeta.VERSION_FILE
        with open(version_file, "w") as fh:
            fh.write("v1.2.3-4-g1234567")

        setup_py = os.path.join(temp, "setup.py")
        meta = SetupMeta().finalize(
            {"_setup_py_path": setup_py, "name": "just-testing", "versioning": "post", "setup_requires": "setupmeta"}
        )

        versioning = meta.versioning
        assert meta.version == "1.2.3.post4"
        assert not versioning.generate_version_file
        assert str(versioning.scm).startswith("snapshot ")
        assert not versioning.scm.is_dirty()
        assert versioning.scm.get_branch() == "HEAD"

        # Trigger artificial rewriting of version file
        versioning.generate_version_file = True
        versioning.auto_fill_version()
        assert version_file.read_text() == "v1.2.3-4-g1234567"


@patch.dict(os.environ, {setupmeta.SCM_DESCRIBE: "1.0"})
def test_find_scm_in_parent():
    with conftest.capture_output():
        meta = new_meta("post")
        versioning = meta.versioning
        assert versioning.enabled
        assert not versioning.problem
        assert setupmeta.project_path() == conftest.TESTS
        assert versioning.scm.root == conftest.TESTS


def check_render(v, expected, main="1.0", distance=None, cid=None, dirty=False):
    version = Version(main=main, distance=distance, commitid=cid, dirty=dirty)
    assert v.strategy.rendered(version) == expected


def test_no_scm(monkeypatch):
    with conftest.capture_output() as logged, patch("setupmeta.model.project_scm", return_value=None):
        fmt = "branch(a,b):{major}.{minor}.{patch}{post}+{.$*FOO*}.{$BAR1*:}{$*BAR2:}{$BAZ:z}{dirty}"
        meta = new_meta(fmt)
        versioning = meta.versioning

        assert "project not under a supported SCM" in logged

        assert versioning.enabled
        assert versioning.problem == "project not under a supported SCM"
        assert meta.version == "0.0.0"
        assert versioning.strategy
        assert versioning.strategy.branches == ["a", "b"]
        assert not versioning.strategy.problem

        assert str(versioning.strategy) == fmt
        assert "BAZ:z" in str(versioning.strategy.extra_bits)

        check_render(versioning, "1.0.0+z")
        check_render(versioning, "1.0.0.post2+z", distance=2)
        check_render(versioning, "1.0.0.post2+z.dirty", distance=2, dirty=True)

        monkeypatch.setenv("TEST_FOO1", "bar")
        monkeypatch.setenv("TEST_FOO2", "baz")
        check_render(versioning, "1.0.0.post2+bar.z.dirty", distance=2, dirty=True)

        with pytest.raises(setupmeta.UsageError, match="project not under a supported SCM"):
            versioning.bump("patch")


@patch.dict(os.environ, {setupmeta.SCM_DESCRIBE: "v1.2.3-4-g1234567-dirty"})
@patch("setupmeta.versioning.find_scm_root", return_value=None)
def test_version_from_env_var(*_):
    with conftest.capture_output():
        meta = new_meta("post")
        versioning = meta.versioning
        assert meta.version == "1.2.3.post4+dirty"
        assert versioning.enabled
        assert not versioning.generate_version_file
        assert not versioning.problem
        assert versioning.scm.is_dirty()


def quick_check(versioning, expected, describe="v0.1.2-5-g123-dirty", compliant=True):
    meta = new_meta(versioning, scm=conftest.MockGit(describe=describe))
    assert meta.version == expected
    if compliant:
        main_part, _, _ = meta.version.partition("+")
        assert str(parse(main_part)) == main_part

    versioning = meta.versioning
    assert versioning.enabled
    assert not versioning.generate_version_file
    assert not versioning.problem
    expected_dirty = "-dirty" in describe
    assert versioning.scm.is_dirty() == expected_dirty


@patch.dict(os.environ, {"BUILD_ID": "543"})
def test_versioning_variants(*_):
    with conftest.capture_output() as logged:
        # Verify that an unfortunate 'v.N.' does not get interpreted as 'vN.'
        quick_check("dev", "0.0.1.dev1", describe="v.28.0-0-gbd750b5")
        quick_check("dev", "0.0.1.dev1+dirty", describe="v.28.0-0-gbd750b5-dirty")

        quick_check("{major}.{minor}", "0.1+dirty")
        quick_check("{major}.{minor}+", "0.1")
        quick_check("{major}.{minor}{dirty}", "0.1+dirty")
        quick_check("{major}.{minor}{dirty}+", "0.1+dirty")
        quick_check("{major}.{minor}", "0.1", describe="v0.1.2-5-g123")
        quick_check("{major}.{minor}+", "0.1", describe="v0.1.2-5-g123")
        quick_check("{major}.{minor}+", "0.1", describe="v0.1.2-5-g123-dirty")

        quick_check("distance", "0.1.5+dirty")
        quick_check("post", "0.1.2.post5+dirty")
        quick_check("dev", "0.1.3.dev5+dirty")
        quick_check("devcommit", "0.1.3.dev5+g123.dirty")

        # Old allowed notations, should remove eventually
        quick_check("dev+build-id", "0.1.3.dev5+h543.g123.dirty")
        quick_check("post+build-id", "0.1.2.post5+h543.g123.dirty")

        # Aliases
        quick_check("default", "0.1.2.post5+dirty")
        quick_check("tag", "0.1.2.post5+dirty")

        # Edge cases
        quick_check("1.2.3", "1.2.3+dirty")
        quick_check("foo", "foo+dirty", compliant=False)

        quick_check("dev+{commitid}{dirty}", "0.1.3.dev5+g123.dirty")
        quick_check("dev+{commitid}{dirty}", "0.1.3.dev0+g123.dirty", describe="v0.1.2-0-g123-dirty")
        quick_check("dev+{commitid}{dirty}", "0.1.2+g123", describe="v0.1.2-0-g123")

        quick_check("dev+devcommit", "0.1.3.dev5+g123.dirty")
        quick_check("dev+devcommit", "0.1.3.dev0+g123.dirty", describe="v0.1.2-0-g123-dirty")
        quick_check("dev+devcommit", "0.1.2", describe="v0.1.2-0-g123")

        quick_check("post+devcommit", "0.1.2.post5+g123.dirty")
        quick_check("post+devcommit", "0.1.2+g123.dirty", describe="v0.1.2-0-g123-dirty")
        quick_check("post+devcommit", "0.1.2", describe="v0.1.2-0-g123")

        quick_check("dev", "0.1.9rc1", describe="v0.1.9-rc.1-0-gebe2789")
        quick_check("devcommit", "0.1.9rc1", describe="v0.1.9-rc.1-0-gebe2789")
        quick_check("post", "0.1.9rc1+dirty", describe="v0.1.9-rc.1-0-gebe2789-dirty")

        quick_check("dev", "0.1.9rc1.dev1", describe="v0.1.9-rc.1-1-gebe2789")
        quick_check("devcommit", "0.1.9rc1.dev1+gebe2789", describe="v0.1.9-rc.1-1-gebe2789")
        quick_check("devcommit", "0.1.9rc1.dev1+gebe2789.dirty", describe="v0.1.9-rc.1-1-gebe2789-dirty")
        quick_check("post", "0.1.9rc1.post1", describe="v0.1.9-rc.1-1-gebe2789")

        quick_check("devcommit", "0.1.3.dev5+g123", describe="v0.1.2-5-g123")
        quick_check("devcommit", "0.1.3.dev5+g123.dirty")
        quick_check("build-id", "0.1.5+h543.g123.dirty")

        quick_check("post", "5.0.0a1.post1", describe="v5.0-a.1-1-gebe2789")
        quick_check("post", "0.0.0.post1", describe="v5.a1rc2-7-gebe2789", compliant=False)
        quick_check("dev", "0.1.0a0.dev8", describe="v0.1.a-8-gebe2789")

        # Patch is not bump-able
        quick_check("dev", "0.1.0rc0.dev5+dirty", describe="v0.1.rc-5-g123-dirty")
        quick_check("dev", "0.1.0rc1.dev5+dirty", describe="v0.1.rc1-5-g123-dirty")
        quick_check("dev", "0.1.0rc1.dev5+dirty", describe="v0.1.rc.1-5-g123-dirty")
        quick_check("dev", "0.1.0rc1.dev5+dirty", describe="v0.1.rc-1-5-g123-dirty")

        # On tag
        quick_check("dev", "0.1.2", describe="v0.1.2-0-g123")
        quick_check("dev", "0.1.3.dev0+dirty", describe="v0.1.2-0-g123-dirty")
        quick_check("devcommit", "0.1.2", describe="v0.1.2")
        quick_check("devcommit", "0.1.2", describe="v0.1.2-0-g123")
        quick_check("devcommit", "0.1.3.dev7+g123", describe="v0.1.2-7-g123")
        quick_check("devcommit", "0.1.3.dev0+g123.dirty", describe="v0.1.2-0-g123-dirty")

        assert "patch version component should be .0" in logged


def test_bump_patch():
    with conftest.capture_output() as logged:
        meta = new_meta("post", scm=conftest.MockGit(describe="v0.1.2.rc-5-g123"))
        versioning = meta.versioning
        versioning.bump("patch")
        assert "Would run: git tag -a v0.1.3" in logged
        assert "Not committing" in logged
        assert "Not pushing" in logged


def test_no_extra():
    with conftest.capture_output() as logged:
        meta = new_meta("{major}.{minor}+", scm=conftest.MockGit())
        versioning = meta.versioning
        check_strategy(versioning, "{major}.{minor}")
        check_render(versioning, "1.0")
        check_render(versioning, "1.0", distance=2)
        check_render(versioning, "1.0", distance=2, dirty=True)

        meta = new_meta("{major}.{minor}.{$FOO}+", scm=conftest.MockGit())
        versioning = meta.versioning
        assert meta.version == "0.1+None"
        check_strategy(versioning, "{major}.{minor}+{$FOO}")
        check_render(versioning, "1.0+None")
        check_render(versioning, "1.0+None", distance=2)
        check_render(versioning, "1.0+None", distance=2, dirty=True)

        assert "patch version component should be .0" in logged


def extra_version(version):
    if version.dirty:
        return "extra"

    if version.distance:
        return "d%s" % version.distance

    return ""


def test_invalid_part():
    with conftest.capture_output() as logged:
        versioning = {"foo": "bar", "main": "{foo}.{major}.{minor}{", "extra": extra_version}
        meta = new_meta(versioning, scm=conftest.MockGit())
        versioning = meta.versioning
        assert "invalid" in str(versioning.strategy.main_bits)
        assert meta.version is None
        assert versioning.problem == "invalid versioning part 'foo'"
        check_strategy(versioning, "{foo}.{major}.{minor}{+function 'extra_version'")
        check_render(versioning, "invalid.1.0")
        check_render(versioning, "invalid.1.0+d2", distance=2)
        check_render(versioning, "invalid.1.0+extra", distance=2, dirty=True)

        assert "Ignored fields for 'versioning': {'foo': 'bar'}" in logged

        with pytest.raises(setupmeta.UsageError, match="invalid versioning part 'foo'"):
            versioning.get_bump("minor")


def test_invalid_main():
    with conftest.capture_output() as logged:
        meta = new_meta({"main": extra_version, "extra": ""}, scm=conftest.MockGit())
        versioning = meta.versioning
        check_strategy(versioning, "function 'extra_version'")
        check_render(versioning, "")
        check_render(versioning, "d2", distance=2)
        check_render(versioning, "extra", distance=2, dirty=True)
        with pytest.raises(setupmeta.UsageError):
            versioning.bump("minor")

        assert "you have pending changes" in logged


def test_malformed():
    with conftest.capture_output():
        meta = new_meta({"main": None, "extra": ""}, name="", scm=conftest.MockGit())
        versioning = meta.versioning
        assert meta.version is None
        assert not versioning.enabled
        assert versioning.problem == "No versioning format specified"


def test_custom_version_tag():
    with conftest.capture_output():
        meta = new_meta({"main": "distance", "extra": "", "version_tag": "v*.*"}, scm=conftest.MockGit(describe="v0.1.2-3-g123"))
        versioning = meta.versioning
        assert versioning.strategy.version_tag == "v*.*"
        assert versioning.scm.version_tag == "v*.*"
        assert str(versioning.scm.get_version()) == "v0.1.2-3-g123"


def test_distance_marker():
    with conftest.capture_output():
        meta = new_meta("{major}.{minor}.{distance}", scm=conftest.MockGit())
        versioning = meta.versioning
        assert versioning.enabled
        assert not versioning.problem
        assert not versioning.strategy.problem
        assert meta.version == "0.1.3+dirty"
        check_strategy(versioning, "{major}.{minor}.{distance}+{dirty}")


def test_preconfigured_build_id():
    """Verify that short notations expand to the expected format"""
    check_preconfigured("{major}.{minor}.{patch}{post}+{dirty}", "post", "default", "tag")
    check_preconfigured("{major}.{minor}.{distance}+{dirty}", "distance")
    check_preconfigured("{major}.{minor}.{distance}+h{$*BUILD_ID:local}.{commitid}{dirty}", "build-id")
    check_preconfigured("{major}.{minor}.{patch}{dev}+h{$*BUILD_ID:local}.{commitid}{dirty}", "dev+build-id")
    check_preconfigured("{major}.{minor}.{patch}{post}+h{$*BUILD_ID:local}.{commitid}{dirty}", "post+build-id")


def check_preconfigured(expected, *shorts):
    with conftest.capture_output():
        for short in shorts:
            meta = new_meta(short, scm=conftest.MockGit())
            versioning = meta.versioning
            assert versioning.enabled
            assert not versioning.problem
            assert not versioning.strategy.problem
            check_strategy(versioning, expected)


@patch.dict(os.environ, {"BUILD_ID": "543"})
def test_preconfigured_strategies(*_):
    with conftest.capture_output():
        check_strategy_distance("v0.1.2-3-g123-dirty")
        check_strategy_distance("v0.1.2-3-g123")
        check_strategy_build_id("v0.1.2-3-g123-dirty")
        check_strategy_build_id("v0.1.2-3-g123")


def check_strategy_distance(describe):
    meta = new_meta("distance", scm=conftest.MockGit(describe=describe))
    versioning = meta.versioning
    assert versioning.enabled
    assert not versioning.problem
    assert not versioning.strategy.problem
    assert "major" in str(versioning.strategy.main_bits)
    assert "dirty" in str(versioning.strategy.extra_bits)
    check_strategy(versioning, "{major}.{minor}.{distance}+{dirty}")
    if "dirty" in describe:
        assert meta.version == "0.1.3+dirty"

        with pytest.raises(setupmeta.UsageError):
            # Can't effectively bump if checkout is dirty
            versioning.bump("minor", commit=True)

    else:
        assert meta.version == "0.1.3"

    with pytest.raises(setupmeta.UsageError):
        # Can't bump 'patch' with 'distance' format
        versioning.bump("patch")

    check_bump(versioning)


def check_strategy_build_id(describe):
    meta = new_meta("build-id", scm=conftest.MockGit(describe=describe))
    versioning = meta.versioning
    assert versioning.enabled
    assert not versioning.problem
    assert not versioning.strategy.problem
    assert "major" in str(versioning.strategy.main_bits)
    assert "commitid" in str(versioning.strategy.extra_bits)
    check_strategy(versioning, "{major}.{minor}.{distance}+h{$*BUILD_ID:local}.{commitid}{dirty}")
    if "dirty" in describe:
        assert meta.version == "0.1.3+h543.g123.dirty"

        with pytest.raises(setupmeta.UsageError):
            # Can't effectively bump when checkout is dirty
            versioning.bump("minor", commit=True)

    else:
        assert meta.version == "0.1.3+h543.g123"

    check_bump(versioning)


def check_bump(versioning):
    with conftest.capture_output() as logged:
        versioning.bump("major")
        assert "Not committing bump, use --commit to commit" in logged
        assert 'git tag -a v1.0.0 -m "Version 1.0.0"' in logged

    with conftest.capture_output() as logged:
        versioning.bump("minor", push=True)
        assert "Not committing bump, use --commit to commit" in logged
        assert 'git tag -a v0.2.0 -m "Version 0.2.0"' in logged
        assert "git push --tags origin" in logged

    with pytest.raises(setupmeta.UsageError):
        versioning.bump("foo")


def test_missing_tags():
    with conftest.capture_output() as logged:
        meta = new_meta("distance", scm=conftest.MockGit(describe="v0.1.2-3-g123", local_tags="v1.0\nv1.1", remote_tags="v1.0\nv2.0"))
        versioning = meta.versioning
        assert versioning.enabled
        assert not versioning.problem
        assert not versioning.strategy.problem
        with pytest.raises(setupmeta.UsageError):
            # Can't effectively bump when remote tags are not all present locally
            versioning.bump("minor", commit=True)
        assert "patch version component should be .0" in logged
