// ****************************************************************************
//
//          Aevol - An in silico experimental evolution platform
//
// ****************************************************************************
//
// Copyright: See the AUTHORS file provided with the package or <www.aevol.fr>
// Web: http://www.aevol.fr/
// E-mail: See <http://www.aevol.fr/contact/>
// Original Authors : Guillaume Beslon, Carole Knibbe, David Parsons
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// ****************************************************************************

#include <getopt.h>

#include <cstdlib>

#include <filesystem>
#include <format>
#include <fstream>

#include "aevol.h"

// Helper functions
void print_help(std::filesystem::path prog_path);
auto interpret_cmd_line_args(int argc, char* argv[]) -> std::tuple<const char*, std::filesystem::path, bool>;

int main(int argc, char* argv[]) {
  // Print the hash and date of the commit used to compile this executable
  std::cout << std::format("Running aevol version {}\n", aevol::version_string);

  const auto [time_or_chkp, output_path, verbose] = interpret_cmd_line_args(argc, argv);

  // Open output file
  auto user_interface_output = std::make_unique<aevol::UserInterfaceOutput>(output_path, 1);

  // Load checkpoint
  auto checkpoint_explorer = aevol::CheckpointExplorer::make_from_checkpoint_time_or_path(time_or_chkp);

  user_interface_output->write_indiv_output(checkpoint_explorer->best_indiv());
  user_interface_output->write_grid_output(
      checkpoint_explorer->grid().width(), checkpoint_explorer->grid().height(), checkpoint_explorer->population());
  user_interface_output->write_environment_output(checkpoint_explorer->target());

  std::cout << std::format("UI outputs written to {}\n", output_path.string());

  return EXIT_SUCCESS;
}

void print_help(std::filesystem::path prog_path) {
  auto prog_name = prog_path.filename().string();

  std::cout << "******************************************************************************\n"
            << "*                                                                            *\n"
            << "*                        aevol - Artificial Evolution                        *\n"
            << "*                                                                            *\n"
            << "* Aevol is a simulation platform that allows one to let populations of       *\n"
            << "* digital organisms evolve in different conditions and study experimentally  *\n"
            << "* the mechanisms responsible for the structuration of the genome and the     *\n"
            << "* transcriptome.                                                             *\n"
            << "*                                                                            *\n"
            << "******************************************************************************\n"
            << "\n"
            << std::format("{}:\nGenerate ui output files from aevol checkpoints.\n", prog_name)
            << "\n"
            << "UI output files are json files that can be used e.g. by the aevol python package to generate "
            << "graphical views of an individual or the population.\n"
            << "The aevol python package can be installed with 'pip install aevol'.\n"
            << "\n"
            << std::format("Usage : {} -h or --help\n", prog_name)
            << std::format("   or : {} -V or --version\n", prog_name)
            << std::format("   or : {} [-v] TIMESTEP\n", prog_name)
            << std::format("   or : {} [-v] PATH_TO_CHECKPOINT\n", prog_name) //##############
            << "\n"
            <<             "   TIMESTEP:             the timestep for which to create the output\n"
            <<             "   PATH_TO_CHECKPOINT:   path to checkpoint file (.ae)\n"
            << "\n"
            << "Options\n"
            << "  -h, --help\n\tprint this help, then exit\n"
            << "  -V, --version\n\tprint version number, then exit\n"
            << "  -v, --verbose\n\tbe verbose\n";
}

auto interpret_cmd_line_args(int argc, char* argv[]) -> std::tuple<const char*, std::filesystem::path, bool> {
  // Command-line option variables
  auto verbose = bool{false};

  // Define allowed options
  const char* options_list = "hVv";
  int option_index = 0;
  static struct option long_options_list[] = {
      {"help",          no_argument,       nullptr, 'h'},
      {"version",       no_argument,       nullptr, 'V'},
      {"verbose",       no_argument,       nullptr, 'v'},
      {0, 0, 0, 0}
  };

  // Get actual values of the CLI options
  int option;
  while ((option = getopt_long(argc, argv, options_list, long_options_list, &option_index)) != -1) {
    switch (option) {
      case 'h' : {
        print_help(argv[0]);
        exit(EXIT_SUCCESS);
      }
      case 'V' : {
        aevol::print_aevol_version();
        exit(EXIT_SUCCESS);
      }
      case 'v' : {
        verbose = true;
        break;
      }
      default : {
        // An error message is printed in getopt_long, we just need to exit
        exit(EXIT_FAILURE);
      }
    }
  }

  // Check number of positional arguments
  if (argc != optind + 1) {
    std::cerr << std::format("{0}: please provide timestep\n"
                             "Try with '--help' for more information.\n", argv[0]);
    exit(EXIT_FAILURE);
  }

  auto output_path = "ui_output_dir";

  return std::make_tuple(argv[optind], output_path, verbose);
}
