/**
 * Copyright (c) FOM-Nikhef 2014-
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2014-
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */

#define _XOPEN_SOURCE	600

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <fnmatch.h>
#include <pwd.h>
#include <ctype.h>

#include <lcmaps/lcmaps_log.h>
#include <lcmaps/lcmaps_defines.h>

#include "lcmaps_gridmapfile.h"


/************************************************************************
 * defines
 ************************************************************************/

#define DELIMITER   ','	    /* delimiter between mappings */

#define LOCAL_GRIDMAP	".gridmap"

#define LOGSTR_PREFIX	"lcmaps_gridmapfile-"


/************************************************************************
 * private prototypes
 ************************************************************************/
 
/* Checks whether the user can switch identity (to root): when either getuid()
 * or geteuid() returns 0, or when seteuid(0) is successful, we can switch.
 * return 1 for normal user, 0 for non-normal user and -1 in case of error
 * (cannot revert to original euid) */ 
static int is_normal_user(void);

/* Reads in the given file into allocated buffer. Buffer needs freeing.
 * needs freeing.
 * return 1 on success, 0 on no gridmapfile but default user, -1 on error */
static int read_mapfile(const char *filename, char **buffer);

/* Find the default grid-mapfile: first try GRIDMAP, GLOBUSMAP, globusmap and
 * GlobusMap env variables. If unset: when normal user: use .gridmap in
 * homedirectory (via getpwuid(getuid())->pw) or current directory. If not
 * normal user: use /etc/grid-security/grid-mapfile
 * default_file needs to be freed.
 * return 0 on success, -1 on error */
static int get_default_mapfile(char **default_file);

/* Find the name of the real user. This is used in case there is no default
 * gridmapdir, and the process is running as normal user (no root, no setuid).
 * return 0 on success, -1 on error */
static int get_self(char **username);

/* Gets next line in buffer and splits out into DN (either between double quoted
 * or white-space terminated) and mappings parts. Next line will be pointed at
 * by next or NULL when no next line. Leading whitespace and terminator
 * characters are skipped from the mappings part and it is truncated at the
 * first comment character. When the mappings part is empty it is set to NULL.
 * Escape sequences in DN part are expanded: \\xHL becomes H*16+L and \\C
 * becomes C. DN must be freed.
 * returns 1 on non-empty DN part, 0 on no DN and -1 on error. */
static int get_line(char *buffer, char **dn, char **mappings, char **next);

/* Finds next mapping in buffer. Buffer should not start with a white space or
 * delimiter. When a mapping is found, next is put to the first character in
 * buffer after the end of the mapping. Escaped characters (such as "\ " in the
 * mapping are expanded. The resulting mapping needs to be freed.
 * return 1 on successful mapping, 0 on no mapping and -1 on error */
static int get_mapping(char *buffer, char **mapping, char **next);


/************************************************************************
 * public functions
 ************************************************************************/

/**
 * Using a grid-mapfile (globusidfile) it looks for a DN or FQAN (globusidp).
 * When options does not contain MATCH_ONLY_DN, it will then look for a mapping
 * which matches searchstr and upon success put the resulting mapping in
 * useridp.
 * When grid-mapfile is unset, a default grid-mapfile is obtained using
 * get_default_mapfile().The default file location is based on environment
 * variables and/or users passwd/ldap entry:
 * 1) GRIDMAP, GLOBUSMAP, globusmap or GlobusMap when set
 * 2a) normal users:
 *  i) users homedir followed by .gridmap 
 *  ii) .gridmap by itself
 * 2b) setuid or root users:
 *  /etc/grid-security/grid-mapfile
 * If the default file does not exist *and* the calling process is non-setuid
 * and non-root, the calling real user account is returned. 
 * return 1 on success, 0 on no match or no mapping and -1 on error.
 */
int lcmaps_gridmapfile(const char *globusidfile, const char *globusidp,
		       const char *searchstr, int options, char **useridp)  {
    const char *mapfile=globusidfile;
    char *default_file=NULL;
    char *filebuffer=NULL, *buffer=NULL, *next_buffer=NULL;
    char *mappings=NULL, *next_mappings=NULL;
    char *found_globusidp=NULL, *mapping=NULL;
    int line,rc;

    /* protect: globusidfile can be NULL, searchstr may be NULL when
     * MATCH_ONLY_DN is set, so check later */
    if (!globusidp || !useridp || (!searchstr && (options & MATCH_ONLY_DN)==0))	{
	lcmaps_log(LOG_ERR,"%s: 1 or more input arguments is NULL.\n",__func__);
	return -1;
    }

    /* Get default if unset */
    if (mapfile==NULL || mapfile[0]=='\0')	{
	if (get_default_mapfile(&default_file)==-1)
	    return -1;
	mapfile=default_file;
    }

    /* log the file */
    lcmaps_log(LOG_DEBUG, "%s: Using %smapfile %s\n",
	    __func__, default_file ? "default " : "", mapfile);

    /* Read in the specified or default grid-mapfile */
    rc=read_mapfile(mapfile, &filebuffer);

    /* Parse return value of read_mapfile */
    if (rc==-1) /* error */
	goto cleanup;
    if (rc==0)	{ /* cannot open grid-mapfile: does not exist */
	/* fail if grid-mapfile is specified or if we require a grid-mapfile or
	 * if we are not a 'normal' user */
	if (globusidfile || (options & REQUIRE_MAPFILE)!=0 ||
	    is_normal_user()!=1)
	{
	    lcmaps_log(LOG_ERR, "%s: cannot open %smapfile %s\n",
		    __func__, default_file ? "default " : "", mapfile);
	    rc=-1;
	    goto cleanup;
	}
	/* Try get ourselves */
	if ( (rc=get_self(useridp))==-1)
	    goto cleanup; /* and return -1 */
	/* Will return our own account: for use with personal Gatekeeper */
	lcmaps_log(LOG_NOTICE,
		"%s: no grid-mapfile: will use own account \"%s\" as default\n",
		__func__, *useridp);
	goto cleanup; /* and return 0 */
    }

    /* loop over all lines */
    for (buffer=filebuffer, line=1; buffer; buffer=next_buffer, line++)	{
	/* free any previous found globusidp */
	free(found_globusidp); found_globusidp=NULL;
	/* get new line */
	rc=get_line(buffer, &found_globusidp, &mappings, &next_buffer);
	if (rc==-1) /* error */
	    goto cleanup;
	if (rc==0)  /* empty DN part */
	    continue;

	/* We have a non-empty DN: match it! */
	if ( (options & MATCH_WILD_CHARS)!=0 )	{
	    if (fnmatch(found_globusidp, globusidp, FNM_NOESCAPE)!=0)
		continue;   /* no match */
	} else	{
	    if (strcmp(globusidp, found_globusidp)!=0)
		continue;   /* no match */
	}

	/* We have a match */
	if (options & MATCH_ONLY_DN) {
	    lcmaps_log(LOG_INFO, "%s: Found match (line %d) \"%s\"\n",
		       __func__, line, found_globusidp);
	    rc=1;   /* found match */
	    goto cleanup;
	}
	/* We need mapping, check they are present */
	if (mappings==NULL) { /* empty mappings */
	    lcmaps_log(LOG_WARNING,
		    "%s: Skipping malformed line %d: no mappings.\n",
		    __func__, line);
	    continue;
	}

	/* Log match on debug */
	lcmaps_log(LOG_DEBUG,"%s: Found match (line %d) %s\n",
		    __func__, line, found_globusidp);

	/* Need to match the mappings */
	do {
	    rc=get_mapping(mappings, &mapping, &next_mappings);
	    if (rc==-1)
		goto cleanup;
	    if (rc==0)	/* empty mapping */
		continue;
	    
	    /* compare the mapping */
	    if (options & MATCH_EXACT)    { /* full string match */
		if (strcmp(mapping, searchstr)==0)	{
		    rc=1;   /* found match */
		    goto cleanup_mapping;
		}
	    } else {
		if (options & MATCH_INCLUDE) { /* substring match */
		    if (strncmp(mapping, searchstr, strlen(searchstr))==0) {
			rc=1;	/* found match */
			goto cleanup_mapping;
		    }
		} else { /* inverted substring match */
		    if (strncmp(mapping, searchstr, strlen(searchstr))!=0) {
			rc=1;	/* found match */
			goto cleanup_mapping;
		    }
		}
	    }
	} while ( (mappings=next_mappings) );

	/* No mapping so far, go to next line */
	lcmaps_log(LOG_DEBUG,
		"%s: No matching mapping found for matching globusidp \"%s\" "
		"(line %d)\n", __func__, found_globusidp, line);
    }

    /* No mapping found */
    rc=0;

cleanup_mapping:
    /* Did we successfully map? */
    if (rc==1)  {
	lcmaps_log(LOG_INFO, "%s: Found mapping %s for \"%s\" (line %d)\n",
		   __func__, mapping, found_globusidp, line);
	*useridp=mapping;
    } else
	free(mapping);
cleanup:

    free(default_file);	    /* default filename */
    free(filebuffer);	    /* grid-mapfile buffer */
    free(found_globusidp);  /* DN */

    /* success or failure */
    return rc;
}


/**
 * Find the properly prefixed mapfile or gridmapdir for relative filenames: when
 * normal user: use directly. If not normal user: use /etc/grid-security/
 * The resulting file will be checked for existence using stat
 * prefixed_file needs to be freed.
 * return 0 on success, -1 on error
 */
int lcmaps_get_prefixed_file(const char *file, char **prefixed_file)	{
    char *path=NULL;
    int len;
    struct stat stat_buf;

    /* protect */
    if (!file || !prefixed_file)    {
	lcmaps_log(LOG_ERR,"%s: 1 or more input arguments is NULL.\n",__func__);
	return -1;
    }

    /* Check whether we somehow can become root, or switch to another
     * account */
    switch (is_normal_user())	{
	case 1:
	    /* we are normal-user: directly use relative path */
	    if ( (path=strdup(file))==NULL )	{
		lcmaps_log(LOG_ERR, "%s: out of memory\n", __func__);
		return -1;
	    }
	    lcmaps_log(LOG_NOTICE,
		    "%s: using relative path \"%s\" for given path.\n",
		    __func__, path);
	    break;
	case 0:
	    /* we are root or can change: build-up secure prefix */
	    len=snprintf(path, 0, SECURITY_DIR"%s", file);
	    if (len<0)  {
		lcmaps_log(LOG_ERR,"%s: snprintf failed: %s\n",
		    __func__, strerror(errno));
		return -1;
	    }
	    if ( (path=malloc((size_t)(len+1)))==NULL )	{
		lcmaps_log(LOG_ERR,"%s: out of memory\n", __func__);
		return -1;
	    }
	    snprintf(path, (size_t)(len+1), SECURITY_DIR"%s", file);
	    break;
	default:
	    return -1;  /* error */
    }

    /* Check existence: check we can stat it */
    if (stat(path, &stat_buf)==-1) {
	lcmaps_log(LOG_ERR,
	    "%s: Cannot stat \"%s\": %s.\n", __func__, path, strerror(errno));
	free(path);
	return -1;
    }

    /* set parameter and return */
    *prefixed_file=path;

    return 0;
}


/************************************************************************
 * private functions
 ************************************************************************/

/**
 * Checks whether the user can switch identity (to root): when either getuid()
 * or geteuid() returns 0, or when seteuid(0) is successful, we can switch.
 * return 1 for normal user, 0 for non-normal user and -1 in case of error
 * (cannot revert to original euid)
 */ 
static int is_normal_user(void)	{
    const char *logstr=LOGSTR_PREFIX"is_normal_user";
    static int init=0;
    static int normal_user=-1;
    int rc;
    uid_t euid,uid;

    if (init)
	return normal_user;

    /* Check whether we somehow can become someone else, or switch to another
     * account */
    rc=-1;
    if ((euid=geteuid())!=(uid=getuid()) || euid==0 || (rc=seteuid(0))==0) {
	normal_user=0;
	/* revert when we got here due to seteuid returning 0 */
	if (rc==0 && (seteuid(euid)<0)) {
	    /* this should not be possible, but handle in any case */
	    lcmaps_log(LOG_ERR,
		       "%s: cannot revert to effective uid %lu: %s\n",
		       logstr, (long unsigned)euid, strerror(errno));
	    normal_user=-1;
	}
    } else
	normal_user=1;

    init=1; /* prevent retrying, even in case of error */
    return normal_user;
}

/**
 * Reads in the given file into allocated buffer. Buffer needs freeing.
 * return 1 on success, 0 on no gridmapfile, -1 on error
 */
static int read_mapfile(const char *filename, char **buffer)	{
    const char *logstr=LOGSTR_PREFIX"read_mapfile";
    char *mybuffer=NULL;
    int fd=-1;
    struct stat stat_buf;
    size_t size;

    /* protect */
    if (!filename || !buffer)  {
	lcmaps_log(LOG_ERR,"%s: 1 or more input arguments is NULL.\n",logstr);
	return -1;
    }

    /* open file */
    if ( (fd=open(filename,O_RDONLY))<0)	{
	if (errno==ENOENT)  {	/* when file does not exist return 0 */
	    lcmaps_log(LOG_DEBUG, "%s: cannot open mapfile %s: %s\n",
		       logstr, filename, strerror(errno));
	    return 0;
	}
	lcmaps_log(LOG_ERR, "%s: cannot open mapfile %s: %s\n",
		       logstr, filename, strerror(errno));
	return -1;
    }

    /* get size */
    if (fstat(fd,&stat_buf)<0)    {
	lcmaps_log(LOG_ERR, "%s: cannot stat mapfile %s: %s\n",
		   logstr, filename, strerror(errno));
	goto error;
    }

    /* allocate buffer: need one extra for \0 byte */
    size=(size_t)stat_buf.st_size;
    if ( (mybuffer=malloc(size+1))==NULL )    {
	lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
	goto error;
    }

    /* Only read in file when there actually is data */
    if (size>0)	{ /* read file */
	if (read(fd,mybuffer,size)!=(ssize_t)size) {
	    lcmaps_log(LOG_ERR, "%s: cannot read mapfile %s: %s\n",
		       logstr, filename, strerror(errno));
	    goto error;
	}
    } else  { /* Probably not what you want... */
	lcmaps_log(LOG_INFO,"%s: mapfile %s has size zero.\n",
		   logstr, filename);
    }
    mybuffer[size]='\0';  /* Need to terminate, read() does not */

    /* all done succesfully */
    close(fd);
    *buffer=mybuffer;

    return 1;

error:
    if (fd>=0)
	close(fd);
    free(mybuffer);

    return -1;
}


/**
 * Find the default grid-mapfile: first try GRIDMAP, GLOBUSMAP, globusmap and
 * GlobusMap env variables. If unset: when normal user: use .gridmap in
 * homedirectory (via getpwuid(getuid())->pw) or current directory. If not
 * normal user: use /etc/grid-security/grid-mapfile
 * default_file needs to be freed.
 * return 0 on success, -1 on error
 */
static int get_default_mapfile(char **default_file)	{
    const char *logstr=LOGSTR_PREFIX"get_default_mapfile";
    char *char_p, *env;
    char *path=NULL;
    int len;
    uid_t uid;
    struct passwd *pw;

    /* protect */
    if (!default_file)	{
	lcmaps_log(LOG_ERR,"%s: input argument is default_file NULL.\n",logstr);
	return -1;
    }

    /* Check gridmap and globusmap env variables */
    if ( ( (env="GRIDMAP",  char_p = getenv(env))!= NULL ||
	   (env="GLOBUSMAP",char_p = getenv(env))!= NULL ||
	   (env="globusmap",char_p = getenv(env))!= NULL ||
	   (env="GlobusMap",char_p = getenv(env))!= NULL ) && char_p[0]!='\0')
    {
	/* found match */
	path=strdup(char_p);
	lcmaps_log(LOG_DEBUG,
		"%s: using grid-mapfile \"%s\" from env variable \"%s\"\n",
		logstr, path, env);
    } else {
	/* Next default depends on identity */
	switch (is_normal_user())   {
	    case 0:
		/* we are root or can change: only use non-user default */
		path=strdup(SECURITY_DIR"grid-mapfile");
		break;
	    case 1:
		/* we're normal user */
		/* Get homedirectory, original Globus (1.0.0) code used home,
		 * Home and HOME, new Globus code (5.2.5) uses pwd from
		 * getpwuid_r, see globus_gsi_sysconfig_get_home_dir_unix in
		 * globus_gsi_system_config.c and
		 * globus_gsi_sysconfig_get_gridmap_filename_unix */
		uid=getuid();
		if ( (pw=getpwuid(uid))==NULL || pw->pw_dir==NULL )   {
		    lcmaps_log(LOG_INFO,
			    "%s: cannot get homedir for uid %d, "
			    "will use relative path for \"%s\"\n",
			    logstr, (int)uid, LOCAL_GRIDMAP);
		    /* no homedir, use directly */
		    path=strdup(LOCAL_GRIDMAP);
		    break;
		}
		/* found homedir: use it to prefix */
		len=snprintf(path,0, "%s/"LOCAL_GRIDMAP, pw->pw_dir);
		if (len<0)  {
		    lcmaps_log(LOG_ERR,"%s: snprintf failed: %s\n",
			logstr, strerror(errno));
		    return -1;
		}
		if ( (path=malloc((size_t)(len+1)))==NULL )	{
		    lcmaps_log(LOG_ERR,"%s: out of memory\n", logstr);
		    return -1;
		}
		snprintf(path,(size_t)(len+1),"%s/"LOCAL_GRIDMAP,
			 pw->pw_dir);
		break;
	    default:
		return -1;  /* error */
	}
    }

    /* any error now is an out of memory */
    if (path==NULL) {
	lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
	return -1;
    }

    /* set parameter and return */
    *default_file=path;

    return 0;
}


/**
 * Find the name of the real user. This is used in case there is no default
 * gridmapdir, and the process is running as normal user (no root, no setuid).
 * return 0 on success, -1 on error
 */
static int get_self(char **username)	{
    const char *logstr=LOGSTR_PREFIX"get_self";
    struct passwd *pw;

    /* protect */
    if (!username)  {
	lcmaps_log(LOG_ERR,"%s: input argument username is NULL.\n",logstr);
	return -1;
    }

    if ( (pw=getpwuid(getuid()))==NULL || pw->pw_name==NULL)	{
	lcmaps_log(LOG_ERR, "%s: cannot determine calling user.\n", logstr);
	return -1;
    }

    if ((*username=strdup(pw->pw_name))==NULL)	{
	lcmaps_log(LOG_ERR,"%s: out of memory\n", logstr);
	return -1;
    }

    return 0;
}


/**
 * Gets next line in buffer and splits out into DN (either between double quoted
 * or white-space terminated) and mappings parts. Next line will be pointed at
 * by next or NULL when no next line. Leading whitespace and terminator
 * characters are skipped from the mappings part and it is truncated at the
 * first comment character. When the mappings part is empty it is set to NULL.
 * Escape sequences in DN part are expanded: \\xHL becomes H*16+L and \\C
 * becomes C. DN must be freed.
 * returns 1 on non-empty DN part, 0 on no DN and -1 on error.
 */
static int get_line(char *buffer, char **dn, char **mappings, char **next)  {
    const char *logstr=LOGSTR_PREFIX"get_line";
    char *pos, *dn_part, *remainder, *dn_tmp;
    int i,dn_len;

    /* protect */
    if (!buffer || !dn || !mappings || !next)	{
	lcmaps_log(LOG_ERR,"%s: 1 or more input arguments is NULL.\n",logstr);
	return -1;
    }

    /* Find end of line and truncate */ 
    if ( (pos=strchr(buffer,'\n')) )	{   /* we have a next line */
	*pos='\0';
	*next=(pos[1] ? pos+1 : NULL);
    } else  /* no more lines after this one */
	*next=NULL;

    /* Find start of DN part: skip leading white-space */
    for (pos=buffer; *pos==' ' || *pos=='\t'; pos++);

    /* Parse first character of DN part */
    if (*pos=='#' || *pos=='\0') /* empty line: no match */
	return 0;

    /* Get DN part */
    if (*pos=='"')  { /* quoted string: find un-escaped end-quote */
	for (i=1; pos[i]; i++)	{
	    if (pos[i]=='\\')	{   /* skip escaped char when applicable */
		if (!pos[i+1])	{
		    lcmaps_log(LOG_ERR,
			    "%s: continuation char not supported\n",logstr);
		    return -1;
		}
		i++;
		continue;
	    }
	    if (pos[i]=='"')
		break;
	}
	/* how did we end? */
	if (!pos[i])   {
	    lcmaps_log(LOG_ERR,"%s: invalid line: quote-inbalance: %s\n",
			logstr, buffer);
	    return -1;
	}
	/* found the end of the DN part: truncate and remove quotes */
	dn_part=pos+1;
	pos[i]='\0';
	remainder=&(pos[i+1]);
    } else {	/* white-space terminated string */
	for (i=0; pos[i]; i++)	{
	    if (pos[i]=='\\')	{  /* skip escaped char when applicable */
		if (!pos[i+1])	{
		    lcmaps_log(LOG_ERR,
			    "%s: continuation char not supported\n",logstr);
		    return -1;
		}
		i++;
		continue;
	    }
	    if (pos[i]=='\t' || pos[i]==' ')
		break;
	}
	/* found the end of the DN part: truncate when needed */
	dn_part=pos;
	if (pos[i]) {
	    pos[i]='\0';
	    remainder=&(pos[i+1]);
	} else
	    remainder=&(pos[i]);
    }
    /* store dn length */
    dn_len=i;

    /* Remove leading white space from remainder */
    while (*remainder==' ' || *remainder=='\t' || *remainder==DELIMITER)
	remainder++;

    /* Remove comments from remainder */
    for (i=0; remainder[i]; i++)    {
	if (remainder[i]=='\\')	{   /* skip escaped char when applicable */
	    if (!remainder[i+1])    {
		lcmaps_log(LOG_ERR,
			"%s: continuation char not supported\n",logstr);
		return -1;
	    }
	    i++;
	    continue;
	}
	if (remainder[i]=='#')	{
	    /* skip rest of remainder, don't try to remove trailing whitespace
	     * here, as it might be escaped in all kinds of complicated ways */
	    remainder[i]='\0';
	    break;
	}
    }

    /* Expand escapes in dn_part */
    if ( (dn_tmp=strdup(dn_part))==NULL )	{
	lcmaps_log(LOG_ERR,"%s: out of memory\n",logstr);
	return -1;
    }
    /* truncate dn_tmp */
    free(*dn); /* free old dn */
    *dn=dn_tmp;
    *mappings=(*remainder ? remainder : NULL);

    /* Is DN empty or not? */
    return (dn_len ? 1 : 0);
}


/**
 * Finds next mapping in buffer. Buffer should not start with a white space or
 * delimiter. When a mapping is found, next is put to the first character in
 * buffer after the end of the mapping. Escaped characters (such as "\ " in the
 * mapping are expanded. The resulting mapping needs to be freed.
 * return 1 on successful mapping, 0 on no mapping and -1 on error
 */
static int get_mapping(char *buffer, char **mapping, char **next)    {
    const char *logstr=LOGSTR_PREFIX"get_mapping";
    char *remainder=NULL,*mapping_tmp;
    int i,j,k,mapping_len;

    /* protect */
    if (!buffer || !mapping || !next)	{
	lcmaps_log(LOG_ERR,"%s: 1 or more input arguments is NULL.\n",logstr);
	return -1;
    }

    /* Find end of mapping and truncate */
    for (i=0; buffer[i]; i++)	{
	if (buffer[i]=='\\')	{  /* skip escaped char when applicable */
	    /* skip escaped char, we already checked continuation characters */
	    i++;
	    continue;
	}
	if (buffer[i]==DELIMITER || buffer[i]=='\t' || buffer[i]==' ')	{
	    /* found end of mapping: truncate */
	    buffer[i]='\0';
	    /* set remainder and skip leading white space and delimiters */
	    for (remainder=&(buffer[i+1]);
		 remainder[0]==' ' || remainder[0]=='\t' ||
		    remainder[0]==DELIMITER;
		 remainder++);
	    break;
	}
    }
    /* store mapping length */
    mapping_len=i;

    /* Now expand escape chars */
    if ( (mapping_tmp=malloc((size_t)(mapping_len+1)))==NULL )	{
	lcmaps_log(LOG_ERR,"%s: out of memory\n",logstr);
	return -1;
    }
    for (j=k=0; buffer[j]; j++)	{
	if (buffer[j]=='\\')	{   /* found escaped char: copy next char */
	    mapping_tmp[k++]=buffer[j+1];
	    j+=1;
	} else { /* normal char */
	    mapping_tmp[k++]=buffer[j];
	}
    }
    /* \0 terminate */
    mapping_tmp[k]='\0';
    free(*mapping); /* free old mapping */
    *mapping=mapping_tmp;
    *next=remainder;	/* either remainder or NULL */

    /* Is mapping empty or not? */
    return (mapping_len ? 1 : 0);
}
