/*
 * Copyright 2015 The Error Prone Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.errorprone.bugpatterns;

import static com.google.common.base.Verify.verify;
import static com.google.errorprone.BugPattern.SeverityLevel.ERROR;
import static com.google.errorprone.util.ASTHelpers.getStartPosition;

import com.google.errorprone.BugPattern;
import com.google.errorprone.VisitorState;
import com.google.errorprone.bugpatterns.BugChecker.MethodInvocationTreeMatcher;
import com.google.errorprone.bugpatterns.BugChecker.NewClassTreeMatcher;
import com.google.errorprone.fixes.Fix;
import com.google.errorprone.fixes.SuggestedFix;
import com.google.errorprone.matchers.Description;
import com.google.errorprone.util.ASTHelpers;
import com.sun.source.tree.MethodInvocationTree;
import com.sun.source.tree.NewClassTree;
import com.sun.source.tree.Tree;
import com.sun.tools.javac.code.Symbol;
import com.sun.tools.javac.code.Symbol.MethodSymbol;
import java.util.List;

/** A {@link BugChecker}; see the associated {@link BugPattern} annotation for details. */
@BugPattern(
    summary = "Non-generic methods should not be invoked with type arguments",
    severity = ERROR)
public class UnnecessaryTypeArgument extends BugChecker
    implements MethodInvocationTreeMatcher, NewClassTreeMatcher {

  @Override
  public Description matchMethodInvocation(MethodInvocationTree tree, VisitorState state) {
    return check(tree, tree.getTypeArguments(), state);
  }

  @Override
  public Description matchNewClass(NewClassTree tree, VisitorState state) {
    return check(tree, tree.getTypeArguments(), state);
  }

  private Description check(Tree tree, List<? extends Tree> arguments, VisitorState state) {
    Symbol sym = ASTHelpers.getSymbol(tree);
    if (!(sym instanceof MethodSymbol methodSymbol)) {
      return Description.NO_MATCH;
    }

    int expected = methodSymbol.getTypeParameters().size();
    int actual = arguments.size();

    if (actual <= expected) {
      return Description.NO_MATCH;
    }

    for (MethodSymbol superMethod : ASTHelpers.findSuperMethods(methodSymbol, state.getTypes())) {
      if (!superMethod.getTypeParameters().isEmpty()) {
        // Exempt methods that override generic methods to preserve the substitutability of the
        // two types.
        return Description.NO_MATCH;
      }
    }

    return describeMatch(tree, buildFix(tree, arguments, state));
  }

  /** Constructor a fix that deletes the set of type arguments. */
  private static Fix buildFix(Tree tree, List<? extends Tree> arguments, VisitorState state) {

    int startAbsolute = getStartPosition(tree);
    int lower = getStartPosition(arguments.getFirst()) - startAbsolute;
    int upper = state.getEndPosition(arguments.getLast()) - startAbsolute;

    String source = state.getSourceForNode(tree);

    while (lower >= 0 && source.charAt(lower) != '<') {
      lower--;
    }
    while (upper < source.length() && source.charAt(upper) != '>') {
      upper++;
    }

    // There's a small chance that the fix will be incorrect because there's a '<' or '>' in
    // a comment (e.g. `this.</*<*/T/*>*/>f()`), it should never be the case that we don't find
    // any angle brackets.
    verify(source.charAt(lower) == '<' && source.charAt(upper) == '>');

    Fix fix = SuggestedFix.replace(startAbsolute + lower, startAbsolute + upper + 1, "");
    return fix;
  }
}
