package Zing::Dropbox;

use 5.014;

use strict;
use warnings;

use registry 'Zing::Types';
use routines;

use Data::Object::Class;
use Data::Object::ClassHas;

extends 'Zing::KeyVal';

our $VERSION = '0.08'; # VERSION

# ATTRIBUTES

has 'name' => (
  is => 'ro',
  isa => 'Str',
  init_arg => undef,
  new => 1,
  mod => 1,
);

fun new_name($self) {
  '$dropbox'
}

# BUILDERS

fun BUILD($self) {
  $self->{name} = '$dropbox';

  return $self;
}

# METHODS

method poll(Str $name, Int $secs) {
  my $data;
  my $time = time + $secs;

  until ($data = $self->recv($name)) {
    last if time >= $time;
  }

  $self->drop($name);

  return $data;
}

1;

=encoding utf8

=head1 NAME

Zing::Dropbox - Transient Store

=cut

=head1 ABSTRACT

Transient Key/Value Store

=cut

=head1 SYNOPSIS

  use Zing::Dropbox;

  my $dropbox = Zing::Dropbox->new;

=cut

=head1 DESCRIPTION

This package provides a general-purpose key/value store for transient data.

=cut

=head1 INHERITS

This package inherits behaviors from:

L<Zing::KeyVal>

=cut

=head1 LIBRARIES

This package uses type constraints from:

L<Zing::Types>

=cut

=head1 METHODS

This package implements the following methods:

=cut

=head2 poll

  poll(Str $name, Int $secs) : Maybe[HashRef]

The poll method watches (i.e. polls) the key specified for C<n> seconds and
either a) returns the data once it's received and drops the key, or b) returns
undefined.

=over 4

=item poll example #1

  # given: synopsis

  $dropbox->poll(ephemeral => 0);

=back

=over 4

=item poll example #2

  # given: synopsis

  # some other process places the data
  $dropbox->send(ephemeral => { data => '...' });

  # poll "ephemeral" for 10 secs, except the data was already placed
  $dropbox->poll(ephemeral => 10);

=back

=over 4

=item poll example #3

  # given: synopsis

  # poll "ephemeral" for 10 secs
  $dropbox->poll(ephemeral => 10);

  # some other process places the data
  $dropbox->send(ephemeral => { data => '...' });

=back

=cut

=head1 AUTHOR

Al Newkirk, C<awncorp@cpan.org>

=head1 LICENSE

Copyright (C) 2011-2019, Al Newkirk, et al.

This is free software; you can redistribute it and/or modify it under the terms
of the The Apache License, Version 2.0, as elucidated in the L<"license
file"|https://github.com/iamalnewkirk/zing/blob/master/LICENSE>.

=head1 PROJECT

L<Wiki|https://github.com/iamalnewkirk/zing/wiki>

L<Project|https://github.com/iamalnewkirk/zing>

L<Initiatives|https://github.com/iamalnewkirk/zing/projects>

L<Milestones|https://github.com/iamalnewkirk/zing/milestones>

L<Contributing|https://github.com/iamalnewkirk/zing/blob/master/CONTRIBUTE.md>

L<Issues|https://github.com/iamalnewkirk/zing/issues>

=cut
