﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/waf-regional/WAFRegionalRequest.h>
#include <aws/waf-regional/WAFRegional_EXPORTS.h>

#include <utility>

namespace Aws {
namespace WAFRegional {
namespace Model {

/**
 */
class AssociateWebACLRequest : public WAFRegionalRequest {
 public:
  AWS_WAFREGIONAL_API AssociateWebACLRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssociateWebACL"; }

  AWS_WAFREGIONAL_API Aws::String SerializePayload() const override;

  AWS_WAFREGIONAL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A unique identifier (ID) for the web ACL. </p>
   */
  inline const Aws::String& GetWebACLId() const { return m_webACLId; }
  inline bool WebACLIdHasBeenSet() const { return m_webACLIdHasBeenSet; }
  template <typename WebACLIdT = Aws::String>
  void SetWebACLId(WebACLIdT&& value) {
    m_webACLIdHasBeenSet = true;
    m_webACLId = std::forward<WebACLIdT>(value);
  }
  template <typename WebACLIdT = Aws::String>
  AssociateWebACLRequest& WithWebACLId(WebACLIdT&& value) {
    SetWebACLId(std::forward<WebACLIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN (Amazon Resource Name) of the resource to be protected, either an
   * application load balancer or Amazon API Gateway stage. </p> <p>The ARN should be
   * in one of the following formats:</p> <ul> <li> <p>For an Application Load
   * Balancer:
   * <code>arn:aws:elasticloadbalancing:<i>region</i>:<i>account-id</i>:loadbalancer/app/<i>load-balancer-name</i>/<i>load-balancer-id</i>
   * </code> </p> </li> <li> <p>For an Amazon API Gateway stage:
   * <code>arn:aws:apigateway:<i>region</i>::/restapis/<i>api-id</i>/stages/<i>stage-name</i>
   * </code> </p> </li> </ul>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  AssociateWebACLRequest& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_webACLId;

  Aws::String m_resourceArn;
  bool m_webACLIdHasBeenSet = false;
  bool m_resourceArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace WAFRegional
}  // namespace Aws
