/*
 * Decompiled with CFR 0.152.
 */
package org.apache.maven.index.updater;

import java.io.BufferedInputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UTFDataFormatException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.zip.GZIPInputStream;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.FieldType;
import org.apache.lucene.index.IndexOptions;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.IndexableField;
import org.apache.lucene.index.IndexableFieldType;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.maven.index.ArtifactInfo;
import org.apache.maven.index.context.DocumentFilter;
import org.apache.maven.index.context.IndexUtils;
import org.apache.maven.index.context.IndexingContext;
import org.apache.maven.index.context.NexusAnalyzer;
import org.apache.maven.index.context.NexusIndexWriter;
import org.apache.maven.index.updater.FSDirectoryFactory;
import org.apache.maven.index.updater.IndexUpdateRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexDataReader {
    private static final Logger LOGGER = LoggerFactory.getLogger(IndexDataReader.class);
    private final DataInputStream dis;
    private final Path tempStorage;
    private final DocumentFilter filter;
    private final FSDirectoryFactory factory;
    private final int threads;

    public IndexDataReader(InputStream is) throws IOException {
        this(is, 1);
    }

    public IndexDataReader(InputStream is, int threads) throws IOException {
        this(is, null, null, null, threads);
    }

    public IndexDataReader(InputStream is, IndexUpdateRequest request) throws IOException {
        this(is, request.getIndexTempDir() != null ? request.getIndexTempDir().toPath() : null, request.getExtractionFilter(), request.getFSDirectoryFactory(), request.getThreads());
    }

    public IndexDataReader(InputStream is, Path tempStorage, DocumentFilter filter, FSDirectoryFactory factory, int threads) throws IOException {
        BufferedInputStream data;
        if (threads < 1) {
            throw new IllegalArgumentException("Reader threads must be greater than zero: " + threads);
        }
        this.tempStorage = Objects.requireNonNullElse(tempStorage, Path.of(System.getProperty("java.io.tmpdir"), new String[0]));
        this.factory = Objects.requireNonNullElse(factory, FSDirectoryFactory.DEFAULT);
        this.filter = filter;
        this.threads = threads;
        is.mark(2);
        if (is.read() == 31 && is.read() == 139) {
            is.reset();
            data = new BufferedInputStream(new GZIPInputStream(is, 8192), 8192);
        } else {
            is.reset();
            data = new BufferedInputStream(is, 8192);
        }
        this.dis = new DataInputStream(data);
    }

    public IndexDataReadResult readIndex(IndexWriter w, IndexingContext context) throws IOException {
        if (this.threads == 1) {
            return this.readIndexST(w, context);
        }
        return this.readIndexMT(w, context);
    }

    private IndexDataReadResult readIndexST(IndexWriter w, IndexingContext context) throws IOException {
        Document doc;
        LOGGER.debug("Reading ST index...");
        Instant start = Instant.now();
        long timestamp = this.readHeader();
        Date date = null;
        if (timestamp != -1L) {
            date = new Date(timestamp);
            IndexUtils.updateTimestamp(w.getDirectory(), date);
        }
        int n = 0;
        HashSet<String> rootGroups = new HashSet<String>();
        HashSet<String> allGroups = new HashSet<String>();
        while ((doc = this.readDocument()) != null) {
            this.addToIndex(doc, context, w, rootGroups, allGroups);
            ++n;
        }
        w.commit();
        IndexDataReadResult result = new IndexDataReadResult();
        result.setDocumentCount(n);
        result.setTimestamp(date);
        result.setRootGroups(rootGroups);
        result.setAllGroups(allGroups);
        LOGGER.debug("Reading ST index done in {} sec", (Object)Duration.between(start, Instant.now()).getSeconds());
        return result;
    }

    private IndexDataReadResult readIndexMT(IndexWriter w, IndexingContext context) throws IOException {
        LOGGER.debug("Reading MT index...");
        Instant start = Instant.now();
        long timestamp = this.readHeader();
        int n = 0;
        Document theEnd = new Document();
        ConcurrentHashMap.KeySetView rootGroups = ConcurrentHashMap.newKeySet();
        ConcurrentHashMap.KeySetView allGroups = ConcurrentHashMap.newKeySet();
        ArrayBlockingQueue<Document> queue = new ArrayBlockingQueue<Document>(10000);
        ExecutorService executorService = Executors.newFixedThreadPool(this.threads);
        List errors = Collections.synchronizedList(new ArrayList());
        ArrayList<FSDirectory> siloDirectories = new ArrayList<FSDirectory>(this.threads);
        ArrayList<IndexWriter> siloWriters = new ArrayList<IndexWriter>(this.threads);
        AtomicBoolean stopEarly = new AtomicBoolean(false);
        LOGGER.debug("Creating {} silo writer threads...", (Object)this.threads);
        for (int i = 0; i < this.threads; ++i) {
            int silo = i;
            FSDirectory siloDirectory = this.tempDirectory("silo" + i);
            siloDirectories.add(siloDirectory);
            siloWriters.add(this.tempWriter(siloDirectory));
            executorService.execute(() -> {
                LOGGER.debug("Starting thread {}", (Object)Thread.currentThread().getName());
                try {
                    try {
                        Document doc;
                        while ((doc = (Document)queue.take()) != theEnd) {
                            this.addToIndex(doc, context, (IndexWriter)siloWriters.get(silo), rootGroups, allGroups);
                        }
                    }
                    catch (Throwable e) {
                        errors.add(e);
                        if (stopEarly.compareAndSet(false, true)) {
                            queue.clear();
                            executorService.shutdownNow();
                        }
                    }
                }
                finally {
                    LOGGER.debug("Done thread {}", (Object)Thread.currentThread().getName());
                }
            });
        }
        LOGGER.debug("Loading up documents into silos");
        try {
            Document doc;
            while (!stopEarly.get() && (doc = this.readDocument()) != null) {
                queue.put(doc);
                ++n;
            }
            LOGGER.debug("Signalling END");
            for (int i = 0; i < this.threads; ++i) {
                queue.put(theEnd);
            }
            LOGGER.debug("Shutting down threads");
            executorService.shutdown();
            executorService.awaitTermination(5L, TimeUnit.MINUTES);
        }
        catch (InterruptedException e) {
            throw new IOException("Interrupted", e);
        }
        if (!errors.isEmpty()) {
            if (errors.stream().allMatch(ex -> ex instanceof IOException || ex instanceof InterruptedException)) {
                IOException exception = new IOException("Error during load of index");
                errors.forEach(exception::addSuppressed);
                throw exception;
            }
            RuntimeException exception = new RuntimeException("Error during load of index");
            errors.forEach(exception::addSuppressed);
            throw exception;
        }
        LOGGER.debug("Silos loaded...");
        Date date = null;
        if (timestamp != -1L) {
            date = new Date(timestamp);
            IndexUtils.updateTimestamp(w.getDirectory(), date);
        }
        LOGGER.debug("Closing silo writers...");
        for (IndexWriter siloWriter : siloWriters) {
            siloWriter.commit();
            siloWriter.close();
        }
        LOGGER.debug("Merging silo directories...");
        w.addIndexes(siloDirectories.toArray(new Directory[0]));
        LOGGER.debug("Cleanup of silo directories...");
        for (FSDirectory siloDirectory : siloDirectories) {
            File dir = siloDirectory.getDirectory().toFile();
            siloDirectory.close();
            IndexUtils.delete(dir);
        }
        LOGGER.debug("Finalizing...");
        w.commit();
        IndexDataReadResult result = new IndexDataReadResult();
        result.setDocumentCount(n);
        result.setTimestamp(date);
        result.setRootGroups(rootGroups);
        result.setAllGroups(allGroups);
        LOGGER.debug("Reading MT index done in {} sec", (Object)Duration.between(start, Instant.now()).getSeconds());
        return result;
    }

    private FSDirectory tempDirectory(String name) throws IOException {
        return this.factory.open(Files.createTempDirectory(this.tempStorage, name + ".dir", new FileAttribute[0]).toFile());
    }

    private IndexWriter tempWriter(FSDirectory directory) throws IOException {
        IndexWriterConfig config = new IndexWriterConfig((Analyzer)new NexusAnalyzer());
        config.setUseCompoundFile(false);
        return new NexusIndexWriter((Directory)directory, config);
    }

    private void addToIndex(Document doc, IndexingContext context, IndexWriter indexWriter, Set<String> rootGroups, Set<String> allGroups) throws IOException {
        ArtifactInfo ai = IndexUtils.constructArtifactInfo(doc, context);
        if (ai != null) {
            if (this.filter == null || this.filter.accept(doc)) {
                indexWriter.addDocument((Iterable)IndexUtils.updateDocument(doc, context, false, ai));
                rootGroups.add(ai.getRootGroup());
                allGroups.add(ai.getGroupId());
            }
        } else if (doc.getField("allGroups") == null && doc.getField("rootGroups") == null) {
            indexWriter.addDocument((Iterable)doc);
        }
    }

    public long readHeader() throws IOException {
        boolean hdrbyte = true;
        if (1 != this.dis.readByte()) {
            throw new IOException("Provided input contains unexpected data (0x01 expected as 1st byte)!");
        }
        return this.dis.readLong();
    }

    public Document readDocument() throws IOException {
        String uinfoString;
        int fieldCount;
        try {
            fieldCount = this.dis.readInt();
        }
        catch (EOFException ex) {
            return null;
        }
        Document doc = new Document();
        for (int i = 0; i < fieldCount; ++i) {
            doc.add((IndexableField)this.readField());
        }
        Field uinfoField = (Field)doc.getField(ArtifactInfo.UINFO);
        String info = doc.get(ArtifactInfo.INFO);
        if (uinfoField != null && info != null && !info.isEmpty() && (uinfoString = uinfoField.stringValue()).endsWith("|NA")) {
            int elem = 0;
            int i = -1;
            while ((i = info.indexOf("|", i + 1)) != -1) {
                if (++elem != 6) continue;
                String extension = info.substring(i + 1);
                uinfoField.setStringValue(uinfoString + "|" + ArtifactInfo.nvl(extension));
                break;
            }
        }
        return doc;
    }

    private Field readField() throws IOException {
        int flags = this.dis.read();
        FieldType fieldType = new FieldType();
        if ((flags & 1) > 0) {
            boolean tokenized = (flags & 2) > 0;
            fieldType.setTokenized(tokenized);
        }
        fieldType.setIndexOptions(IndexOptions.DOCS_AND_FREQS_AND_POSITIONS);
        fieldType.setStored((flags & 4) > 0);
        String name = this.dis.readUTF();
        String value = IndexDataReader.readUTF(this.dis);
        return new Field(name, (CharSequence)value, (IndexableFieldType)fieldType);
    }

    private static String readUTF(DataInput in) throws IOException {
        int c;
        int count;
        char[] chararr;
        byte[] bytearr;
        int utflen = in.readInt();
        try {
            bytearr = new byte[utflen];
            chararr = new char[utflen];
        }
        catch (OutOfMemoryError e) {
            throw new IOException("Index data content is inappropriate (is junk?), leads to OutOfMemoryError! See MINDEXER-28 for more information!", e);
        }
        int chararrCount = 0;
        in.readFully(bytearr, 0, utflen);
        for (count = 0; count < utflen && (c = bytearr[count] & 0xFF) <= 127; ++count) {
            chararr[chararrCount++] = (char)c;
        }
        block8: while (count < utflen) {
            c = bytearr[count] & 0xFF;
            switch (c >> 4) {
                case 0: 
                case 1: 
                case 2: 
                case 3: 
                case 4: 
                case 5: 
                case 6: 
                case 7: {
                    ++count;
                    chararr[chararrCount++] = (char)c;
                    continue block8;
                }
                case 12: 
                case 13: {
                    if ((count += 2) > utflen) {
                        throw new UTFDataFormatException("malformed input: partial character at end");
                    }
                    byte char2 = bytearr[count - 1];
                    if ((char2 & 0xC0) != 128) {
                        throw new UTFDataFormatException("malformed input around byte " + count);
                    }
                    chararr[chararrCount++] = (char)((c & 0x1F) << 6 | char2 & 0x3F);
                    continue block8;
                }
                case 14: {
                    if ((count += 3) > utflen) {
                        throw new UTFDataFormatException("malformed input: partial character at end");
                    }
                    byte char2 = bytearr[count - 2];
                    byte char3 = bytearr[count - 1];
                    if ((char2 & 0xC0) != 128 || (char3 & 0xC0) != 128) {
                        throw new UTFDataFormatException("malformed input around byte " + (count - 1));
                    }
                    chararr[chararrCount++] = (char)((c & 0xF) << 12 | (char2 & 0x3F) << 6 | char3 & 0x3F);
                    continue block8;
                }
            }
            throw new UTFDataFormatException("malformed input around byte " + count);
        }
        return new String(chararr, 0, chararrCount);
    }

    public IndexDataReadResult readIndex(IndexDataReadVisitor visitor, IndexingContext context) throws IOException {
        Document doc;
        this.dis.readByte();
        long timestamp = this.dis.readLong();
        Date date = null;
        if (timestamp != -1L) {
            date = new Date(timestamp);
        }
        int n = 0;
        while ((doc = this.readDocument()) != null) {
            visitor.visitDocument(IndexUtils.updateDocument(doc, context, false));
            ++n;
        }
        IndexDataReadResult result = new IndexDataReadResult();
        result.setDocumentCount(n);
        result.setTimestamp(date);
        return result;
    }

    public static class IndexDataReadResult {
        private Date timestamp;
        private int documentCount;
        private Set<String> rootGroups;
        private Set<String> allGroups;

        public void setDocumentCount(int documentCount) {
            this.documentCount = documentCount;
        }

        public int getDocumentCount() {
            return this.documentCount;
        }

        public void setTimestamp(Date timestamp) {
            this.timestamp = timestamp;
        }

        public Date getTimestamp() {
            return this.timestamp;
        }

        public void setRootGroups(Set<String> rootGroups) {
            this.rootGroups = rootGroups;
        }

        public Set<String> getRootGroups() {
            return this.rootGroups;
        }

        public void setAllGroups(Set<String> allGroups) {
            this.allGroups = allGroups;
        }

        public Set<String> getAllGroups() {
            return this.allGroups;
        }
    }

    public static interface IndexDataReadVisitor {
        public void visitDocument(Document var1);
    }
}

