﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/qbusiness/QBusiness_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace QBusiness {
namespace Model {

/**
 * <p>Information about the OAuth 2.0 authentication credential/token used to
 * configure a plugin.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/qbusiness-2023-11-27/OAuth2ClientCredentialConfiguration">AWS
 * API Reference</a></p>
 */
class OAuth2ClientCredentialConfiguration {
 public:
  AWS_QBUSINESS_API OAuth2ClientCredentialConfiguration() = default;
  AWS_QBUSINESS_API OAuth2ClientCredentialConfiguration(Aws::Utils::Json::JsonView jsonValue);
  AWS_QBUSINESS_API OAuth2ClientCredentialConfiguration& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_QBUSINESS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The ARN of the Secrets Manager secret that stores the OAuth 2.0
   * credentials/token used for plugin configuration.</p>
   */
  inline const Aws::String& GetSecretArn() const { return m_secretArn; }
  inline bool SecretArnHasBeenSet() const { return m_secretArnHasBeenSet; }
  template <typename SecretArnT = Aws::String>
  void SetSecretArn(SecretArnT&& value) {
    m_secretArnHasBeenSet = true;
    m_secretArn = std::forward<SecretArnT>(value);
  }
  template <typename SecretArnT = Aws::String>
  OAuth2ClientCredentialConfiguration& WithSecretArn(SecretArnT&& value) {
    SetSecretArn(std::forward<SecretArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of an IAM role used by Amazon Q Business to access the OAuth 2.0
   * authentication credentials stored in a Secrets Manager secret.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  OAuth2ClientCredentialConfiguration& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The redirect URL required by the OAuth 2.0 protocol for Amazon Q Business to
   * authenticate a plugin user through a third party authentication server.</p>
   */
  inline const Aws::String& GetAuthorizationUrl() const { return m_authorizationUrl; }
  inline bool AuthorizationUrlHasBeenSet() const { return m_authorizationUrlHasBeenSet; }
  template <typename AuthorizationUrlT = Aws::String>
  void SetAuthorizationUrl(AuthorizationUrlT&& value) {
    m_authorizationUrlHasBeenSet = true;
    m_authorizationUrl = std::forward<AuthorizationUrlT>(value);
  }
  template <typename AuthorizationUrlT = Aws::String>
  OAuth2ClientCredentialConfiguration& WithAuthorizationUrl(AuthorizationUrlT&& value) {
    SetAuthorizationUrl(std::forward<AuthorizationUrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URL required by the OAuth 2.0 protocol to exchange an end user
   * authorization code for an access token.</p>
   */
  inline const Aws::String& GetTokenUrl() const { return m_tokenUrl; }
  inline bool TokenUrlHasBeenSet() const { return m_tokenUrlHasBeenSet; }
  template <typename TokenUrlT = Aws::String>
  void SetTokenUrl(TokenUrlT&& value) {
    m_tokenUrlHasBeenSet = true;
    m_tokenUrl = std::forward<TokenUrlT>(value);
  }
  template <typename TokenUrlT = Aws::String>
  OAuth2ClientCredentialConfiguration& WithTokenUrl(TokenUrlT&& value) {
    SetTokenUrl(std::forward<TokenUrlT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_secretArn;

  Aws::String m_roleArn;

  Aws::String m_authorizationUrl;

  Aws::String m_tokenUrl;
  bool m_secretArnHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_authorizationUrlHasBeenSet = false;
  bool m_tokenUrlHasBeenSet = false;
};

}  // namespace Model
}  // namespace QBusiness
}  // namespace Aws
