#define _GNU_SOURCE
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"
#include <string.h>

/* Portable memmem - use system version if available, else our own */
#ifndef HAVE_MEMMEM
#if defined(__GLIBC__) || defined(__APPLE__) || defined(__FreeBSD__) || defined(__OpenBSD__) || defined(__NetBSD__)
#define HAVE_MEMMEM 1
#endif
#endif

#if HAVE_MEMMEM
#define util_memmem memmem
#else
static void *util_memmem(const void *haystack, size_t haystacklen,
                         const void *needle, size_t needlelen) {
    if (needlelen == 0) return (void*)haystack;
    if (needlelen > haystacklen) return NULL;
    
    const char *h = (const char*)haystack;
    const char *n = (const char*)needle;
    const char *end = h + haystacklen - needlelen + 1;
    char first = *n;
    
    for (; h < end; h++) {
        if (*h == first && memcmp(h, n, needlelen) == 0) {
            return (void*)h;
        }
    }
    return NULL;
}
#endif

/* ============================================
   Custom op structures
   ============================================ */

static XOP identity_xop;
static XOP always_xop;
static XOP clamp_xop;
static XOP nvl_xop;
static XOP coalesce_xop;

/* Type predicate custom ops - blazing fast, single SV flag check */
static XOP is_ref_xop;
static XOP is_array_xop;
static XOP is_hash_xop;
static XOP is_code_xop;
static XOP is_defined_xop;

/* String predicate custom ops - direct SvPV/SvCUR access */
static XOP is_empty_xop;
static XOP starts_with_xop;
static XOP ends_with_xop;
/* Boolean/Truthiness custom ops - fast truth checks */
static XOP is_true_xop;
static XOP is_false_xop;
static XOP bool_xop;

/* Extended type predicate custom ops */
static XOP is_num_xop;
static XOP is_int_xop;
static XOP is_blessed_xop;
static XOP is_scalar_ref_xop;
static XOP is_regex_xop;
static XOP is_glob_xop;
static XOP is_string_xop;

/* Numeric predicate custom ops */
static XOP is_positive_xop;
static XOP is_negative_xop;
static XOP is_zero_xop;

/* Numeric utility custom ops */
static XOP is_even_xop;
static XOP is_odd_xop;
static XOP is_between_xop;

/* Collection custom ops - direct AvFILL/HvKEYS access */
static XOP is_empty_array_xop;
static XOP is_empty_hash_xop;
static XOP array_len_xop;
static XOP hash_size_xop;
static XOP array_first_xop;
static XOP array_last_xop;

/* String manipulation custom ops */
static XOP trim_xop;
static XOP ltrim_xop;
static XOP rtrim_xop;

/* Conditional custom ops */
static XOP maybe_xop;

/* Numeric custom ops */
static XOP sign_xop;
static XOP min2_xop;
static XOP max2_xop;

/* ============================================
   Memoization structures
   ============================================ */

typedef struct {
    SV *func;           /* Original coderef */
    HV *cache;          /* Result cache */
    IV hits;            /* Cache hits (stats) */
    IV misses;          /* Cache misses (stats) */
} MemoizedFunc;

static MemoizedFunc *g_memos = NULL;
static IV g_memo_size = 0;
static IV g_memo_count = 0;

/* ============================================
   Lazy evaluation structures
   ============================================ */

typedef struct {
    SV *thunk;          /* Deferred computation (coderef) */
    SV *value;          /* Cached result */
    bool forced;        /* Has been evaluated? */
} LazyValue;

static LazyValue *g_lazies = NULL;
static IV g_lazy_size = 0;
static IV g_lazy_count = 0;

/* ============================================
   Always (constant) structures
   ============================================ */

static SV **g_always_values = NULL;
static IV g_always_size = 0;
static IV g_always_count = 0;

/* ============================================
   Once (execute once) structures
   ============================================ */

typedef struct {
    SV *func;           /* Original function */
    SV *result;         /* Cached result */
    bool called;        /* Has been called? */
} OnceFunc;

static OnceFunc *g_onces = NULL;
static IV g_once_size = 0;
static IV g_once_count = 0;

/* ============================================
   Partial application structures
   ============================================ */

typedef struct {
    SV *func;           /* Original function */
    AV *bound_args;     /* Pre-bound arguments */
} PartialFunc;

static PartialFunc *g_partials = NULL;
static IV g_partial_size = 0;
static IV g_partial_count = 0;

/* ============================================
   Forward declarations
   ============================================ */

static XS(xs_memo_call);
static XS(xs_compose_call);
static XS(xs_always_call);
static XS(xs_negate_call);
static XS(xs_once_call);
static XS(xs_partial_call);

/* ============================================
   Utility functions
   ============================================ */

static void ensure_memo_capacity(IV needed) {
    if (needed >= g_memo_size) {
        IV new_size = g_memo_size ? g_memo_size * 2 : 16;
        while (new_size <= needed) new_size *= 2;
        Renew(g_memos, new_size, MemoizedFunc);
        g_memo_size = new_size;
    }
}

static void ensure_lazy_capacity(IV needed) {
    if (needed >= g_lazy_size) {
        IV new_size = g_lazy_size ? g_lazy_size * 2 : 16;
        while (new_size <= needed) new_size *= 2;
        Renew(g_lazies, new_size, LazyValue);
        g_lazy_size = new_size;
    }
}

static void ensure_always_capacity(IV needed) {
    if (needed >= g_always_size) {
        IV new_size = g_always_size ? g_always_size * 2 : 16;
        while (new_size <= needed) new_size *= 2;
        Renew(g_always_values, new_size, SV*);
        g_always_size = new_size;
    }
}

static void ensure_once_capacity(IV needed) {
    if (needed >= g_once_size) {
        IV new_size = g_once_size ? g_once_size * 2 : 16;
        while (new_size <= needed) new_size *= 2;
        Renew(g_onces, new_size, OnceFunc);
        g_once_size = new_size;
    }
}

static void ensure_partial_capacity(IV needed) {
    if (needed >= g_partial_size) {
        IV new_size = g_partial_size ? g_partial_size * 2 : 16;
        while (new_size <= needed) new_size *= 2;
        Renew(g_partials, new_size, PartialFunc);
        g_partial_size = new_size;
    }
}

/* Build cache key from stack arguments */
static SV* build_cache_key(pTHX_ SV **args, IV count) {
    SV *key = newSVpvs("");
    IV i;
    for (i = 0; i < count; i++) {
        if (i > 0) sv_catpvs(key, "\x00");
        if (SvOK(args[i])) {
            STRLEN len;
            const char *pv = SvPV(args[i], len);
            sv_catpvn(key, pv, len);
        } else {
            sv_catpvs(key, "\x01UNDEF\x01");
        }
    }
    return key;
}

/* ============================================
   Custom OP implementations - fastest path
   ============================================ */

/* identity: just return the top of stack */
static OP* pp_identity(pTHX) {
    /* Value already on stack, nothing to do */
    return NORMAL;
}

/* always: push stored value from op_targ index */
static OP* pp_always(pTHX) {
    dSP;
    IV idx = PL_op->op_targ;
    XPUSHs(g_always_values[idx]);
    RETURN;
}

/* clamp: 3 values on stack, return clamped */
static OP* pp_clamp(pTHX) {
    dSP;
    SV *max_sv = POPs;
    SV *min_sv = POPs;
    SV *val_sv = TOPs;  /* Peek, will replace */

    NV value = SvNV(val_sv);
    NV min = SvNV(min_sv);
    NV max = SvNV(max_sv);

    if (value < min) {
        SETs(min_sv);
    } else if (value > max) {
        SETs(max_sv);
    }
    /* else TOPs already has value */

    RETURN;
}

/* nvl: 2 values on stack, return first if defined */
static OP* pp_nvl(pTHX) {
    dSP;
    SV *def_sv = POPs;
    SV *val_sv = TOPs;

    if (!SvOK(val_sv)) {
        SETs(def_sv);
    }
    RETURN;
}

/* ============================================
   Type predicate custom ops - blazing fast!
   These are the fastest possible type checks:
   single SV flag check, no function call overhead
   ============================================ */

/* is_ref: check if value is a reference */
static OP* pp_is_ref(pTHX) {
    dSP;
    SV *sv = TOPs;
    SETs(SvROK(sv) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* is_array: check if value is an arrayref */
static OP* pp_is_array(pTHX) {
    dSP;
    SV *sv = TOPs;
    SETs((SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVAV) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* is_hash: check if value is a hashref */
static OP* pp_is_hash(pTHX) {
    dSP;
    SV *sv = TOPs;
    SETs((SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVHV) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* is_code: check if value is a coderef */
static OP* pp_is_code(pTHX) {
    dSP;
    SV *sv = TOPs;
    SETs((SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVCV) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* is_defined: check if value is defined */
static OP* pp_is_defined(pTHX) {
    dSP;
    SV *sv = TOPs;
    SETs(SvOK(sv) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* ============================================
   String predicate custom ops - blazing fast!
   Direct SvPV/SvCUR access, no function overhead
   ============================================ */

/* is_empty: check if string is undefined or empty */
static OP* pp_is_empty(pTHX) {
    dSP;
    SV *sv = TOPs;
    /* Empty if: undefined OR length is 0 */
    if (!SvOK(sv)) {
        SETs(&PL_sv_yes);
    } else {
        STRLEN len;
        SvPV(sv, len);
        SETs(len == 0 ? &PL_sv_yes : &PL_sv_no);
    }
    RETURN;
}

/* starts_with: check if string starts with prefix */
static OP* pp_starts_with(pTHX) {
    dSP;
    SV *prefix_sv = POPs;
    SV *str_sv = TOPs;

    if (!SvOK(str_sv) || !SvOK(prefix_sv)) {
        SETs(&PL_sv_no);
        RETURN;
    }

    STRLEN str_len, prefix_len;
    const char *str = SvPV(str_sv, str_len);
    const char *prefix = SvPV(prefix_sv, prefix_len);

    if (prefix_len > str_len) {
        SETs(&PL_sv_no);
    } else if (prefix_len == 0) {
        SETs(&PL_sv_yes);  /* Empty prefix always matches */
    } else {
        SETs(memcmp(str, prefix, prefix_len) == 0 ? &PL_sv_yes : &PL_sv_no);
    }
    RETURN;
}

/* ends_with: check if string ends with suffix */
static OP* pp_ends_with(pTHX) {
    dSP;
    SV *suffix_sv = POPs;
    SV *str_sv = TOPs;

    if (!SvOK(str_sv) || !SvOK(suffix_sv)) {
        SETs(&PL_sv_no);
        RETURN;
    }

    STRLEN str_len, suffix_len;
    const char *str = SvPV(str_sv, str_len);
    const char *suffix = SvPV(suffix_sv, suffix_len);

    if (suffix_len > str_len) {
        SETs(&PL_sv_no);
    } else if (suffix_len == 0) {
        SETs(&PL_sv_yes);  /* Empty suffix always matches */
    } else {
        const char *str_end = str + str_len - suffix_len;
        SETs(memcmp(str_end, suffix, suffix_len) == 0 ? &PL_sv_yes : &PL_sv_no);
    }
    RETURN;
}

/* ============================================
   Boolean/Truthiness custom ops - blazing fast!
   Direct SvTRUE check, minimal overhead
   ============================================ */

/* is_true: check if value is truthy (Perl truth semantics) */
static OP* pp_is_true(pTHX) {
    dSP;
    SV *sv = TOPs;
    SETs(SvTRUE(sv) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* is_false: check if value is falsy (Perl truth semantics) */
static OP* pp_is_false(pTHX) {
    dSP;
    SV *sv = TOPs;
    SETs(SvTRUE(sv) ? &PL_sv_no : &PL_sv_yes);
    RETURN;
}

/* bool: normalize to boolean (1 or empty string) */
static OP* pp_bool(pTHX) {
    dSP;
    SV *sv = TOPs;
    SETs(SvTRUE(sv) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* ============================================
   Extended type predicate custom ops - blazing fast!
   ============================================ */

/* is_num: check if value is numeric (has numeric value or looks like number) */
static OP* pp_is_num(pTHX) {
    dSP;
    SV *sv = TOPs;
    /* SvNIOK: has numeric (NV or IV) value cached */
    /* Also check looks_like_number for strings that can be numbers */
    SETs((SvNIOK(sv) || looks_like_number(sv)) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* is_int: check if value is an integer */
static OP* pp_is_int(pTHX) {
    dSP;
    SV *sv = TOPs;
    /* SvIOK: has integer value cached */
    if (SvIOK(sv)) {
        SETs(&PL_sv_yes);
    } else if (SvNOK(sv)) {
        /* It's a float - check if it's a whole number */
        NV nv = SvNV(sv);
        SETs((nv == (NV)(IV)nv) ? &PL_sv_yes : &PL_sv_no);
    } else if (looks_like_number(sv)) {
        /* String that looks like a number - check if integer */
        STRLEN len;
        const char *pv = SvPV(sv, len);
        /* Simple check: no decimal point or exponent */
        bool has_dot = FALSE;
        STRLEN i;
        for (i = 0; i < len; i++) {
            if (pv[i] == '.' || pv[i] == 'e' || pv[i] == 'E') {
                has_dot = TRUE;
                break;
            }
        }
        if (has_dot) {
            /* Has decimal - check if value is actually integer */
            NV nv = SvNV(sv);
            SETs((nv == (NV)(IV)nv) ? &PL_sv_yes : &PL_sv_no);
        } else {
            SETs(&PL_sv_yes);
        }
    } else {
        SETs(&PL_sv_no);
    }
    RETURN;
}

/* is_blessed: check if value is a blessed reference */
static OP* pp_is_blessed(pTHX) {
    dSP;
    SV *sv = TOPs;
    SETs(sv_isobject(sv) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* is_scalar_ref: check if value is a scalar reference (not array/hash/code/etc) */
static OP* pp_is_scalar_ref(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvROK(sv)) {
        SV *rv = SvRV(sv);
        svtype type = SvTYPE(rv);
        /* Scalar refs are < SVt_PVAV (array) */
        SETs((type < SVt_PVAV) ? &PL_sv_yes : &PL_sv_no);
    } else {
        SETs(&PL_sv_no);
    }
    RETURN;
}

/* is_regex: check if value is a compiled regex */
static OP* pp_is_regex(pTHX) {
    dSP;
    SV *sv = TOPs;
    /* SvRXOK: check if SV is a regex (qr//) - available since Perl 5.10 */
    SETs(SvRXOK(sv) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* is_glob: check if value is a glob (*foo) */
static OP* pp_is_glob(pTHX) {
    dSP;
    SV *sv = TOPs;
    SETs((SvTYPE(sv) == SVt_PVGV) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* is_string: check if value is a plain scalar (defined, not a reference) */
static OP* pp_is_string(pTHX) {
    dSP;
    SV *sv = TOPs;
    SETs((SvOK(sv) && !SvROK(sv)) ? &PL_sv_yes : &PL_sv_no);
    RETURN;
}

/* ============================================
   Numeric predicate custom ops - blazing fast!
   Direct SvNV comparison, minimal overhead
   ============================================ */

/* is_positive: check if value is > 0 */
static OP* pp_is_positive(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvNIOK(sv) || looks_like_number(sv)) {
        NV nv = SvNV(sv);
        SETs((nv > 0) ? &PL_sv_yes : &PL_sv_no);
    } else {
        SETs(&PL_sv_no);
    }
    RETURN;
}

/* is_negative: check if value is < 0 */
static OP* pp_is_negative(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvNIOK(sv) || looks_like_number(sv)) {
        NV nv = SvNV(sv);
        SETs((nv < 0) ? &PL_sv_yes : &PL_sv_no);
    } else {
        SETs(&PL_sv_no);
    }
    RETURN;
}

/* is_zero: check if value is == 0 */
static OP* pp_is_zero(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvNIOK(sv) || looks_like_number(sv)) {
        NV nv = SvNV(sv);
        SETs((nv == 0) ? &PL_sv_yes : &PL_sv_no);
    } else {
        SETs(&PL_sv_no);
    }
    RETURN;
}

/* ============================================
   Numeric utility custom ops
   ============================================ */

/* is_even: check if integer is even (single bitwise AND) */
static OP* pp_is_even(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvIOK(sv)) {
        SETs((SvIVX(sv) & 1) == 0 ? &PL_sv_yes : &PL_sv_no);
    } else if (SvNIOK(sv)) {
        NV nv = SvNV(sv);
        if (nv == (NV)(IV)nv) {
            SETs(((IV)nv & 1) == 0 ? &PL_sv_yes : &PL_sv_no);
        } else {
            SETs(&PL_sv_no);
        }
    } else if (looks_like_number(sv)) {
        SETs((SvIV(sv) & 1) == 0 ? &PL_sv_yes : &PL_sv_no);
    } else {
        SETs(&PL_sv_no);
    }
    RETURN;
}

/* is_odd: check if integer is odd (single bitwise AND) */
static OP* pp_is_odd(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvIOK(sv)) {
        SETs((SvIVX(sv) & 1) == 1 ? &PL_sv_yes : &PL_sv_no);
    } else if (SvNIOK(sv)) {
        NV nv = SvNV(sv);
        if (nv == (NV)(IV)nv) {
            SETs(((IV)nv & 1) == 1 ? &PL_sv_yes : &PL_sv_no);
        } else {
            SETs(&PL_sv_no);
        }
    } else if (looks_like_number(sv)) {
        SETs((SvIV(sv) & 1) == 1 ? &PL_sv_yes : &PL_sv_no);
    } else {
        SETs(&PL_sv_no);
    }
    RETURN;
}

/* is_between: check if value is between min and max (inclusive) */
static OP* pp_is_between(pTHX) {
    dSP;
    SV *max_sv = POPs;
    SV *min_sv = POPs;
    SV *val_sv = TOPs;

    if (SvNIOK(val_sv) || looks_like_number(val_sv)) {
        NV val = SvNV(val_sv);
        NV min = SvNV(min_sv);
        NV max = SvNV(max_sv);
        SETs((val >= min && val <= max) ? &PL_sv_yes : &PL_sv_no);
    } else {
        SETs(&PL_sv_no);
    }
    RETURN;
}

/* ============================================
   Collection custom ops - direct AvFILL/HvKEYS access
   ============================================ */

/* is_empty_array: check if arrayref is empty - direct AvFILL */
static OP* pp_is_empty_array(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVAV) {
        AV *av = (AV*)SvRV(sv);
        SETs(AvFILL(av) < 0 ? &PL_sv_yes : &PL_sv_no);
    } else {
        SETs(&PL_sv_no);  /* Not an arrayref */
    }
    RETURN;
}

/* is_empty_hash: check if hashref is empty - direct HvKEYS */
static OP* pp_is_empty_hash(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVHV) {
        HV *hv = (HV*)SvRV(sv);
        SETs(HvKEYS(hv) == 0 ? &PL_sv_yes : &PL_sv_no);
    } else {
        SETs(&PL_sv_no);  /* Not a hashref */
    }
    RETURN;
}

/* array_len: get array length - direct AvFILL + 1 */
static OP* pp_array_len(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVAV) {
        AV *av = (AV*)SvRV(sv);
        SV *len = sv_2mortal(newSViv(AvFILL(av) + 1));
        SETs(len);
    } else {
        SETs(&PL_sv_undef);  /* Not an arrayref */
    }
    RETURN;
}

/* hash_size: get hash key count - direct HvKEYS */
static OP* pp_hash_size(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVHV) {
        HV *hv = (HV*)SvRV(sv);
        SV *size = sv_2mortal(newSViv(HvKEYS(hv)));
        SETs(size);
    } else {
        SETs(&PL_sv_undef);  /* Not a hashref */
    }
    RETURN;
}

/* array_first: get first element without slice overhead */
static OP* pp_array_first(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVAV) {
        AV *av = (AV*)SvRV(sv);
        if (AvFILL(av) >= 0) {
            SV **elem = av_fetch(av, 0, 0);
            SETs(elem ? *elem : &PL_sv_undef);
        } else {
            SETs(&PL_sv_undef);  /* Empty array */
        }
    } else {
        SETs(&PL_sv_undef);  /* Not an arrayref */
    }
    RETURN;
}

/* array_last: get last element without slice overhead */
static OP* pp_array_last(pTHX) {
    dSP;
    SV *sv = TOPs;
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVAV) {
        AV *av = (AV*)SvRV(sv);
        IV last_idx = AvFILL(av);
        if (last_idx >= 0) {
            SV **elem = av_fetch(av, last_idx, 0);
            SETs(elem ? *elem : &PL_sv_undef);
        } else {
            SETs(&PL_sv_undef);  /* Empty array */
        }
    } else {
        SETs(&PL_sv_undef);  /* Not an arrayref */
    }
    RETURN;
}


/* ============================================
   String manipulation custom ops
   ============================================ */

/* trim: remove leading/trailing whitespace */
static OP* pp_trim(pTHX) {
    dSP;
    SV *sv = TOPs;

    if (!SvOK(sv)) {
        SETs(&PL_sv_undef);
        RETURN;
    }

    STRLEN len;
    const char *str = SvPV(sv, len);
    const char *start = str;
    const char *end = str + len;

    /* Skip leading whitespace */
    while (start < end && isSPACE(*start)) {
        start++;
    }

    /* Skip trailing whitespace */
    while (end > start && isSPACE(*(end - 1))) {
        end--;
    }

    /* Create new SV with trimmed content */
    SV *result = sv_2mortal(newSVpvn(start, end - start));
    SETs(result);
    RETURN;
}

/* ltrim: remove leading whitespace only */
static OP* pp_ltrim(pTHX) {
    dSP;
    SV *sv = TOPs;

    if (!SvOK(sv)) {
        SETs(&PL_sv_undef);
        RETURN;
    }

    STRLEN len;
    const char *str = SvPV(sv, len);
    const char *start = str;
    const char *end = str + len;

    /* Skip leading whitespace */
    while (start < end && isSPACE(*start)) {
        start++;
    }

    SV *result = sv_2mortal(newSVpvn(start, end - start));
    SETs(result);
    RETURN;
}

/* rtrim: remove trailing whitespace only */
static OP* pp_rtrim(pTHX) {
    dSP;
    SV *sv = TOPs;

    if (!SvOK(sv)) {
        SETs(&PL_sv_undef);
        RETURN;
    }

    STRLEN len;
    const char *str = SvPV(sv, len);
    const char *end = str + len;

    /* Skip trailing whitespace */
    while (end > str && isSPACE(*(end - 1))) {
        end--;
    }

    SV *result = sv_2mortal(newSVpvn(str, end - str));
    SETs(result);
    RETURN;
}

/* ============================================
   Conditional custom ops
   ============================================ */

/* maybe: return $then if $val is defined, else undef */
static OP* pp_maybe(pTHX) {
    dSP;
    SV *then_sv = POPs;
    SV *val_sv = TOPs;

    if (SvOK(val_sv)) {
        SETs(then_sv);
    } else {
        SETs(&PL_sv_undef);
    }
    RETURN;
}

/* ============================================
   Numeric custom ops
   ============================================ */

/* sign: return -1, 0, or 1 based on value */
static OP* pp_sign(pTHX) {
    dSP;
    SV *sv = TOPs;

    if (!SvNIOK(sv) && !looks_like_number(sv)) {
        SETs(&PL_sv_undef);
        RETURN;
    }

    NV nv = SvNV(sv);
    if (nv > 0) {
        SETs(sv_2mortal(newSViv(1)));
    } else if (nv < 0) {
        SETs(sv_2mortal(newSViv(-1)));
    } else {
        SETs(sv_2mortal(newSViv(0)));
    }
    RETURN;
}

/* min2: return smaller of two values */
static OP* pp_min2(pTHX) {
    dSP;
    SV *b_sv = POPs;
    SV *a_sv = TOPs;

    NV a = SvNV(a_sv);
    NV b = SvNV(b_sv);

    SETs(a <= b ? a_sv : b_sv);
    RETURN;
}

/* max2: return larger of two values */
static OP* pp_max2(pTHX) {
    dSP;
    SV *b_sv = POPs;
    SV *a_sv = TOPs;

    NV a = SvNV(a_sv);
    NV b = SvNV(b_sv);

    SETs(a >= b ? a_sv : b_sv);
    RETURN;
}


/* ============================================
   Call checkers - replace function calls with custom ops
   ============================================ */

/* identity call checker - replaces identity($x) with just $x */
static OP* identity_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    OP *pushop, *argop, *cvop;

    PERL_UNUSED_ARG(namegv);
    PERL_UNUSED_ARG(ckobj);

    /* Get the argument list */
    pushop = cUNOPx(entersubop)->op_first;
    if (!OpHAS_SIBLING(pushop)) {
        pushop = cUNOPx(pushop)->op_first;
    }

    /* Find first real arg (skip pushmark) */
    argop = OpSIBLING(pushop);

    /* Find the cv op (last sibling) */
    cvop = argop;
    while (OpHAS_SIBLING(cvop)) {
        cvop = OpSIBLING(cvop);
    }

    /* Check for exactly one argument */
    if (argop != cvop && OpSIBLING(argop) == cvop) {
        /* Single arg - just return the arg itself */
        OP *arg = argop;

        /* Detach arg from list */
        OpMORESIB_set(pushop, cvop);
        OpLASTSIB_set(arg, NULL);

        op_free(entersubop);
        return arg;  /* Just return the argument op directly! */
    }

    /* Fall through to XS for edge cases */
    return entersubop;
}

/* clamp call checker - replaces clamp($v, $min, $max) with custom op */
static OP* clamp_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    OP *pushop, *arg1, *arg2, *arg3, *cvop;

    PERL_UNUSED_ARG(namegv);
    PERL_UNUSED_ARG(ckobj);

    /* Get the argument list */
    pushop = cUNOPx(entersubop)->op_first;
    if (!OpHAS_SIBLING(pushop)) {
        pushop = cUNOPx(pushop)->op_first;
    }

    /* Find args (skip pushmark) */
    arg1 = OpSIBLING(pushop);  /* value */
    if (!arg1) return entersubop;

    arg2 = OpSIBLING(arg1);    /* min */
    if (!arg2) return entersubop;

    arg3 = OpSIBLING(arg2);    /* max */
    if (!arg3) return entersubop;

    cvop = OpSIBLING(arg3);    /* cv op (should be last) */
    if (!cvop || OpHAS_SIBLING(cvop)) return entersubop;

    /* Build custom op with 3 args on stack */
    OP *newop = newUNOP(OP_NULL, 0, NULL);
    newop->op_type = OP_CUSTOM;
    newop->op_ppaddr = pp_clamp;
    newop->op_flags = OPf_WANT_SCALAR;

    /* Detach args from list */
    OpMORESIB_set(pushop, cvop);
    OpLASTSIB_set(arg3, NULL);
    OpMORESIB_set(arg2, arg3);
    OpMORESIB_set(arg1, arg2);

    /* Build list op to evaluate args */
    OP *listop = op_prepend_elem(OP_LIST, arg1, NULL);
    listop = op_append_elem(OP_LIST, listop, arg2);
    listop = op_append_elem(OP_LIST, listop, arg3);
    listop = op_append_elem(OP_LIST, listop, newop);

    op_free(entersubop);
    return listop;
}

/* Generic call checker for single-arg type predicates */
static OP* type_predicate_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj, OP* (*pp_func)(pTHX)) {
    OP *pushop, *argop, *cvop;

    PERL_UNUSED_ARG(namegv);
    PERL_UNUSED_ARG(ckobj);

    /* Get the argument list */
    pushop = cUNOPx(entersubop)->op_first;
    if (!OpHAS_SIBLING(pushop)) {
        pushop = cUNOPx(pushop)->op_first;
    }

    /* Find first real arg (skip pushmark) */
    argop = OpSIBLING(pushop);

    /* Find the cv op (last sibling) */
    cvop = argop;
    while (OpHAS_SIBLING(cvop)) {
        cvop = OpSIBLING(cvop);
    }

    /* Check for exactly one argument */
    if (argop != cvop && OpSIBLING(argop) == cvop) {
        OP *arg = argop;

        /* Detach arg from list */
        OpMORESIB_set(pushop, cvop);
        OpLASTSIB_set(arg, NULL);

        /* Create custom op as BASEOP (no children - reads from stack) */
        OP *newop = newOP(OP_NULL, 0);
        newop->op_type = OP_CUSTOM;
        newop->op_ppaddr = pp_func;
        newop->op_flags = OPf_WANT_SCALAR;

        /* Build list: arg evaluates first, pushes to stack, then custom op reads it */
        OP *listop = op_append_elem(OP_LIST, arg, newop);

        op_free(entersubop);
        return listop;
    }

    /* Fall through to XS for edge cases */
    return entersubop;
}

/* Individual call checkers for each type predicate */
static OP* is_ref_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_ref);
}

static OP* is_array_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_array);
}

static OP* is_hash_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_hash);
}

static OP* is_code_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_code);
}

static OP* is_defined_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_defined);
}

/* String predicate call checkers */
static OP* is_empty_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_empty);
}

/* Generic two-arg string predicate call checker */
static OP* two_arg_string_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj, OP* (*pp_func)(pTHX)) {
    OP *pushop, *arg1, *arg2, *cvop;

    PERL_UNUSED_ARG(namegv);
    PERL_UNUSED_ARG(ckobj);

    /* Get the argument list */
    pushop = cUNOPx(entersubop)->op_first;
    if (!OpHAS_SIBLING(pushop)) {
        pushop = cUNOPx(pushop)->op_first;
    }

    /* Find args (skip pushmark) */
    arg1 = OpSIBLING(pushop);  /* string */
    if (!arg1) return entersubop;

    arg2 = OpSIBLING(arg1);    /* prefix/suffix */
    if (!arg2) return entersubop;

    cvop = OpSIBLING(arg2);    /* cv op (should be last) */
    if (!cvop || OpHAS_SIBLING(cvop)) return entersubop;

    /* Detach args from the entersub tree */
    OpMORESIB_set(pushop, cvop);

    /* Chain arg1 -> arg2 */
    OpMORESIB_set(arg1, arg2);
    OpLASTSIB_set(arg2, NULL);

    /* 
     * Create a custom BINOP-style op.
     * Use newBINOP to create a proper binary op structure where
     * both arguments are children. The optimizer won't eliminate
     * children of an op that's going to use them.
     */
    OP *binop = newBINOP(OP_NULL, 0, arg1, arg2);
    binop->op_type = OP_CUSTOM;
    binop->op_ppaddr = pp_func;
    binop->op_flags = OPf_WANT_SCALAR | OPf_KIDS | OPf_STACKED;

    op_free(entersubop);
    return binop;
}

static OP* starts_with_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return two_arg_string_call_checker(aTHX_ entersubop, namegv, ckobj, pp_starts_with);
}

static OP* ends_with_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return two_arg_string_call_checker(aTHX_ entersubop, namegv, ckobj, pp_ends_with);
}

/* Boolean/Truthiness call checkers */
static OP* is_true_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_true);
}

static OP* is_false_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_false);
}

static OP* bool_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_bool);
}

/* Extended type predicate call checkers */
static OP* is_num_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_num);
}

static OP* is_int_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_int);
}

static OP* is_blessed_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_blessed);
}

static OP* is_scalar_ref_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_scalar_ref);
}

static OP* is_regex_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_regex);
}

static OP* is_glob_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_glob);
}

static OP* is_string_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_string);
}

/* Numeric predicate call checkers */
static OP* is_positive_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_positive);
}

static OP* is_negative_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_negative);
}

static OP* is_zero_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_zero);
}

/* Numeric utility call checkers */
static OP* is_even_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_even);
}

static OP* is_odd_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_odd);
}

/* is_between needs 3 args - use same pattern as clamp */
static OP* is_between_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    OP *pushop, *arg1, *arg2, *arg3, *cvop;

    PERL_UNUSED_ARG(namegv);
    PERL_UNUSED_ARG(ckobj);

    pushop = cUNOPx(entersubop)->op_first;
    if (!OpHAS_SIBLING(pushop)) {
        pushop = cUNOPx(pushop)->op_first;
    }

    arg1 = OpSIBLING(pushop);
    if (!arg1) return entersubop;

    arg2 = OpSIBLING(arg1);
    if (!arg2) return entersubop;

    arg3 = OpSIBLING(arg2);
    if (!arg3) return entersubop;

    cvop = OpSIBLING(arg3);
    if (!cvop || OpHAS_SIBLING(cvop)) return entersubop;

    OP *newop = newUNOP(OP_NULL, 0, NULL);
    newop->op_type = OP_CUSTOM;
    newop->op_ppaddr = pp_is_between;
    newop->op_flags = OPf_WANT_SCALAR;

    OpMORESIB_set(pushop, cvop);
    OpLASTSIB_set(arg3, NULL);
    OpMORESIB_set(arg2, arg3);
    OpMORESIB_set(arg1, arg2);

    OP *listop = op_prepend_elem(OP_LIST, arg1, NULL);
    listop = op_append_elem(OP_LIST, listop, arg2);
    listop = op_append_elem(OP_LIST, listop, arg3);
    listop = op_append_elem(OP_LIST, listop, newop);

    op_free(entersubop);
    return listop;
}

/* Collection call checkers */
static OP* is_empty_array_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_empty_array);
}

static OP* is_empty_hash_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_is_empty_hash);
}

static OP* array_len_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_array_len);
}

static OP* hash_size_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_hash_size);
}

static OP* array_first_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_array_first);
}

static OP* array_last_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_array_last);
}

/* trim uses single-arg pattern */
static OP* trim_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_trim);
}

static OP* ltrim_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_ltrim);
}

static OP* rtrim_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_rtrim);
}

/* maybe uses two-arg pattern */
static OP* maybe_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return two_arg_string_call_checker(aTHX_ entersubop, namegv, ckobj, pp_maybe);
}

/* Numeric ops */
static OP* sign_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return type_predicate_call_checker(aTHX_ entersubop, namegv, ckobj, pp_sign);
}

static OP* min2_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return two_arg_string_call_checker(aTHX_ entersubop, namegv, ckobj, pp_min2);
}

static OP* max2_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    return two_arg_string_call_checker(aTHX_ entersubop, namegv, ckobj, pp_max2);
}

/* ============================================
   Memo implementation
   ============================================ */

static XS(xs_memo) {
    dXSARGS;
    if (items != 1) croak("Usage: util::memo(\\&func)");

    SV *func = ST(0);
    if (!SvROK(func) || SvTYPE(SvRV(func)) != SVt_PVCV) {
        croak("util::memo requires a coderef");
    }

    /* Allocate memo slot */
    IV idx = g_memo_count++;
    ensure_memo_capacity(idx);

    MemoizedFunc *mf = &g_memos[idx];
    mf->func = SvREFCNT_inc_simple_NN(func);
    mf->cache = newHV();
    mf->hits = 0;
    mf->misses = 0;

    /* Create wrapper CV */
    CV *wrapper = newXS(NULL, xs_memo_call, __FILE__);
    CvXSUBANY(wrapper).any_iv = idx;

    ST(0) = newRV_noinc((SV*)wrapper);
    XSRETURN(1);
}

static XS(xs_memo_call) {
    dXSARGS;
    IV idx = CvXSUBANY(cv).any_iv;
    MemoizedFunc *mf = &g_memos[idx];

    /* Build cache key from arguments */
    SV *key = build_cache_key(aTHX_ &ST(0), items);
    STRLEN key_len;
    const char *key_pv = SvPV(key, key_len);

    /* Check cache */
    SV **cached = hv_fetch(mf->cache, key_pv, key_len, 0);
    if (cached && SvOK(*cached)) {
        mf->hits++;
        SvREFCNT_dec_NN(key);
        if (SvROK(*cached) && SvTYPE(SvRV(*cached)) == SVt_PVAV) {
            AV *av = (AV*)SvRV(*cached);
            IV len = av_len(av) + 1;
            IV i;
            EXTEND(SP, len);
            for (i = 0; i < len; i++) {
                SV **elem = av_fetch(av, i, 0);
                ST(i) = elem ? *elem : &PL_sv_undef;
            }
            XSRETURN(len);
        } else {
            ST(0) = *cached;
            XSRETURN(1);
        }
    }

    mf->misses++;

    ENTER;
    SAVETMPS;
    PUSHMARK(SP);

    IV i;
    EXTEND(SP, items);
    for (i = 0; i < items; i++) {
        PUSHs(ST(i));
    }
    PUTBACK;

    IV count = call_sv(mf->func, G_ARRAY);

    SPAGAIN;

    if (count == 1) {
        SV *result = SvREFCNT_inc(POPs);
        hv_store(mf->cache, key_pv, key_len, result, 0);
        PUTBACK;
        FREETMPS;
        LEAVE;
        SvREFCNT_dec_NN(key);
        ST(0) = result;
        XSRETURN(1);
    } else if (count > 0) {
        AV *av = newAV();
        av_extend(av, count - 1);
        for (i = count - 1; i >= 0; i--) {
            av_store(av, i, SvREFCNT_inc(POPs));
        }
        SV *result = newRV_noinc((SV*)av);
        hv_store(mf->cache, key_pv, key_len, result, 0);
        PUTBACK;
        FREETMPS;
        LEAVE;
        SvREFCNT_dec_NN(key);
        for (i = 0; i < count; i++) {
            SV **elem = av_fetch(av, i, 0);
            ST(i) = elem ? *elem : &PL_sv_undef;
        }
        XSRETURN(count);
    } else {
        hv_store(mf->cache, key_pv, key_len, &PL_sv_undef, 0);
        PUTBACK;
        FREETMPS;
        LEAVE;
        SvREFCNT_dec_NN(key);
        XSRETURN_EMPTY;
    }
}

/* ============================================
   Pipe/Compose implementation
   ============================================ */

static XS(xs_pipe) {
    dXSARGS;
    if (items < 2) croak("Usage: util::pipeline($value, \\&fn1, \\&fn2, ...)");

    SV *value = ST(0);
    IV i;

    for (i = 1; i < items; i++) {
        SV *func = ST(i);
        if (!SvROK(func) || SvTYPE(SvRV(func)) != SVt_PVCV) {
            croak("util::pipeline: argument %d is not a coderef", (int)i);
        }

        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        XPUSHs(value);
        PUTBACK;

        call_sv(func, G_SCALAR);

        SPAGAIN;
        value = POPs;
        SvREFCNT_inc(value);
        PUTBACK;
        FREETMPS;
        LEAVE;
    }

    ST(0) = value;
    XSRETURN(1);
}

static XS(xs_compose) {
    dXSARGS;
    if (items < 1) croak("Usage: util::compose(\\&fn1, \\&fn2, ...)");

    AV *funcs = newAV();
    av_extend(funcs, items - 1);
    IV i;
    for (i = 0; i < items; i++) {
        SV *func = ST(i);
        if (!SvROK(func) || SvTYPE(SvRV(func)) != SVt_PVCV) {
            croak("util::compose: argument %d is not a coderef", (int)(i+1));
        }
        av_store(funcs, i, SvREFCNT_inc_simple_NN(func));
    }

    CV *wrapper = newXS(NULL, xs_compose_call, __FILE__);
    CvXSUBANY(wrapper).any_ptr = (void*)funcs;

    ST(0) = newRV_noinc((SV*)wrapper);
    XSRETURN(1);
}

static XS(xs_compose_call) {
    dXSARGS;
    AV *funcs = (AV*)CvXSUBANY(cv).any_ptr;
    IV func_count = av_len(funcs) + 1;

    SV *value = NULL;

    IV i;
    for (i = func_count - 1; i >= 0; i--) {
        SV **func_ptr = av_fetch(funcs, i, 0);
        if (!func_ptr) continue;

        ENTER;
        SAVETMPS;
        PUSHMARK(SP);

        if (i == func_count - 1) {
            IV j;
            EXTEND(SP, items);
            for (j = 0; j < items; j++) {
                PUSHs(ST(j));
            }
        } else {
            XPUSHs(value);
        }
        PUTBACK;

        call_sv(*func_ptr, G_SCALAR);

        SPAGAIN;
        value = POPs;
        SvREFCNT_inc(value);
        PUTBACK;
        FREETMPS;
        LEAVE;
    }

    ST(0) = value ? value : &PL_sv_undef;
    XSRETURN(1);
}

/* ============================================
   Lazy evaluation implementation
   ============================================ */

static XS(xs_lazy) {
    dXSARGS;
    if (items != 1) croak("Usage: util::lazy(sub { ... })");

    SV *thunk = ST(0);
    if (!SvROK(thunk) || SvTYPE(SvRV(thunk)) != SVt_PVCV) {
        croak("util::lazy requires a coderef");
    }

    IV idx = g_lazy_count++;
    ensure_lazy_capacity(idx);

    LazyValue *lv = &g_lazies[idx];
    lv->thunk = SvREFCNT_inc_simple_NN(thunk);
    lv->value = NULL;
    lv->forced = FALSE;

    SV *obj = newSViv(idx);
    SV *ref = newRV_noinc(obj);
    sv_bless(ref, gv_stashpv("util::Lazy", GV_ADD));

    ST(0) = ref;
    XSRETURN(1);
}

static XS(xs_force) {
    dXSARGS;
    if (items != 1) croak("Usage: util::force($lazy)");

    SV *lazy = ST(0);

    if (!SvROK(lazy) || !sv_derived_from(lazy, "util::Lazy")) {
        ST(0) = lazy;
        XSRETURN(1);
    }

    IV idx = SvIV(SvRV(lazy));
    if (idx < 0 || idx >= g_lazy_count) {
        croak("util::force: invalid lazy value");
    }

    LazyValue *lv = &g_lazies[idx];

    if (lv->forced) {
        ST(0) = lv->value;
        XSRETURN(1);
    }

    ENTER;
    SAVETMPS;
    PUSHMARK(SP);
    PUTBACK;

    call_sv(lv->thunk, G_SCALAR);

    SPAGAIN;
    lv->value = SvREFCNT_inc(POPs);
    lv->forced = TRUE;
    PUTBACK;
    FREETMPS;
    LEAVE;

    SvREFCNT_dec(lv->thunk);
    lv->thunk = NULL;

    ST(0) = lv->value;
    XSRETURN(1);
}

/* ============================================
   Safe navigation (dig) implementation
   ============================================ */

static XS(xs_dig) {
    dXSARGS;
    if (items < 2) croak("Usage: util::dig($hash, @keys)");

    SV *current = ST(0);
    IV i;

    for (i = 1; i < items; i++) {
        if (!SvROK(current) || SvTYPE(SvRV(current)) != SVt_PVHV) {
            XSRETURN_UNDEF;
        }

        HV *hv = (HV*)SvRV(current);
        SV *key = ST(i);
        STRLEN key_len;
        const char *key_pv = SvPV(key, key_len);

        SV **val = hv_fetch(hv, key_pv, key_len, 0);
        if (!val || !SvOK(*val)) {
            XSRETURN_UNDEF;
        }

        current = *val;
    }

    ST(0) = current;
    XSRETURN(1);
}

/* ============================================
   Tap implementation
   ============================================ */

static XS(xs_tap) {
    dXSARGS;
    if (items != 2) croak("Usage: util::tap(\\&block, $value)");

    SV *func = ST(0);
    SV *value = ST(1);

    if (!SvROK(func) || SvTYPE(SvRV(func)) != SVt_PVCV) {
        croak("util::tap: first argument must be a coderef");
    }

    ENTER;
    SAVETMPS;
    SAVE_DEFSV;
    DEFSV_set(value);

    PUSHMARK(SP);
    XPUSHs(value);
    PUTBACK;

    call_sv(func, G_DISCARD | G_VOID);

    SPAGAIN;
    FREETMPS;
    LEAVE;

    ST(0) = value;
    XSRETURN(1);
}

/* ============================================
   Clamp XS fallback
   ============================================ */

static XS(xs_clamp) {
    dXSARGS;
    if (items != 3) croak("Usage: util::clamp($value, $min, $max)");

    NV value = SvNV(ST(0));
    NV min = SvNV(ST(1));
    NV max = SvNV(ST(2));

    if (value < min) {
        ST(0) = ST(1);
    } else if (value > max) {
        ST(0) = ST(2);
    }

    XSRETURN(1);
}

/* ============================================
   Identity XS fallback
   ============================================ */

static XS(xs_identity) {
    dXSARGS;
    if (items != 1) croak("Usage: util::identity($value)");
    XSRETURN(1);
}

/* ============================================
   Always implementation
   ============================================ */

static XS(xs_always) {
    dXSARGS;
    if (items != 1) croak("Usage: util::always($value)");

    IV idx = g_always_count++;
    ensure_always_capacity(idx);

    g_always_values[idx] = SvREFCNT_inc_simple_NN(ST(0));

    CV *wrapper = newXS(NULL, xs_always_call, __FILE__);
    CvXSUBANY(wrapper).any_iv = idx;

    ST(0) = newRV_noinc((SV*)wrapper);
    XSRETURN(1);
}

static XS(xs_always_call) {
    dXSARGS;
    PERL_UNUSED_VAR(items);
    IV idx = CvXSUBANY(cv).any_iv;

    ST(0) = g_always_values[idx];
    XSRETURN(1);
}

/* ============================================
   Stub/noop functions - return constants
   ============================================ */

/* pp_noop - custom op that returns undef */
static OP* pp_noop(pTHX) {
    dSP;
    XPUSHs(&PL_sv_undef);
    RETURN;
}

/* noop call checker - replace with ultra-fast custom op */
static OP* noop_call_checker(pTHX_ OP *entersubop, GV *namegv, SV *ckobj) {
    OP *newop;
    PERL_UNUSED_ARG(namegv);
    PERL_UNUSED_ARG(ckobj);

    op_free(entersubop);

    NewOp(1101, newop, 1, OP);
    newop->op_type = OP_CUSTOM;
    newop->op_ppaddr = pp_noop;
    newop->op_flags = OPf_WANT_SCALAR;
    newop->op_next = newop;

    return newop;
}

/* noop() - does nothing, returns undef. Ignores all arguments. */
static XS(xs_noop) {
    dXSARGS;
    PERL_UNUSED_VAR(items);
    XSRETURN_UNDEF;
}

/* stub_true() - always returns true (1) */
static XS(xs_stub_true) {
    dXSARGS;
    PERL_UNUSED_VAR(items);
    XSRETURN_YES;
}

/* stub_false() - always returns false ('') */
static XS(xs_stub_false) {
    dXSARGS;
    PERL_UNUSED_VAR(items);
    XSRETURN_NO;
}

/* stub_array() - always returns new empty arrayref [] */
static XS(xs_stub_array) {
    dXSARGS;
    PERL_UNUSED_VAR(items);
    ST(0) = sv_2mortal(newRV_noinc((SV*)newAV()));
    XSRETURN(1);
}

/* stub_hash() - always returns new empty hashref {} */
static XS(xs_stub_hash) {
    dXSARGS;
    PERL_UNUSED_VAR(items);
    ST(0) = sv_2mortal(newRV_noinc((SV*)newHV()));
    XSRETURN(1);
}

/* stub_string() - always returns empty string '' */
static XS(xs_stub_string) {
    dXSARGS;
    PERL_UNUSED_VAR(items);
    /* Return shared empty string constant - XSRETURN_NO returns '' */
    XSRETURN_NO;
}

/* stub_zero() - always returns 0 */
static XS(xs_stub_zero) {
    dXSARGS;
    PERL_UNUSED_VAR(items);
    /* Return shared 0 SV */
    ST(0) = &PL_sv_zero;
    XSRETURN(1);
}

/* ============================================
   Functional combinators
   ============================================ */

/* negate(\&pred) - returns a function that returns the opposite */
static XS(xs_negate) {
    dXSARGS;
    if (items != 1) croak("Usage: util::negate(\\&predicate)");

    SV *pred = ST(0);
    if (!SvROK(pred) || SvTYPE(SvRV(pred)) != SVt_PVCV) {
        croak("util::negate: argument must be a coderef");
    }

    CV *wrapper = newXS(NULL, xs_negate_call, __FILE__);
    CvXSUBANY(wrapper).any_ptr = SvREFCNT_inc_simple_NN(pred);

    ST(0) = newRV_noinc((SV*)wrapper);
    XSRETURN(1);
}

static XS(xs_negate_call) {
    dXSARGS;
    SV *pred = (SV*)CvXSUBANY(cv).any_ptr;

    ENTER;
    SAVETMPS;
    PUSHMARK(SP);

    IV i;
    EXTEND(SP, items);
    for (i = 0; i < items; i++) {
        PUSHs(ST(i));
    }
    PUTBACK;

    call_sv(pred, G_SCALAR);

    SPAGAIN;
    SV *result = POPs;
    bool val = SvTRUE(result);
    PUTBACK;
    FREETMPS;
    LEAVE;

    ST(0) = val ? &PL_sv_no : &PL_sv_yes;
    XSRETURN(1);
}

/* once(\&f) - execute once, cache forever */
static XS(xs_once) {
    dXSARGS;
    if (items != 1) croak("Usage: util::once(\\&func)");

    SV *func = ST(0);
    if (!SvROK(func) || SvTYPE(SvRV(func)) != SVt_PVCV) {
        croak("util::once: argument must be a coderef");
    }

    IV idx = g_once_count++;
    ensure_once_capacity(idx);

    OnceFunc *of = &g_onces[idx];
    of->func = SvREFCNT_inc_simple_NN(func);
    of->result = NULL;
    of->called = FALSE;

    CV *wrapper = newXS(NULL, xs_once_call, __FILE__);
    CvXSUBANY(wrapper).any_iv = idx;

    ST(0) = newRV_noinc((SV*)wrapper);
    XSRETURN(1);
}

static XS(xs_once_call) {
    dXSARGS;
    PERL_UNUSED_VAR(items);
    IV idx = CvXSUBANY(cv).any_iv;
    OnceFunc *of = &g_onces[idx];

    if (of->called) {
        ST(0) = of->result ? of->result : &PL_sv_undef;
        XSRETURN(1);
    }

    ENTER;
    SAVETMPS;
    PUSHMARK(SP);
    PUTBACK;

    call_sv(of->func, G_SCALAR);

    SPAGAIN;
    of->result = SvREFCNT_inc(POPs);
    of->called = TRUE;
    PUTBACK;
    FREETMPS;
    LEAVE;

    /* Free the original function, no longer needed */
    SvREFCNT_dec(of->func);
    of->func = NULL;

    ST(0) = of->result;
    XSRETURN(1);
}

/* partial(\&f, @bound) - bind first N args */
static XS(xs_partial) {
    dXSARGS;
    if (items < 1) croak("Usage: util::partial(\\&func, @bound_args)");

    SV *func = ST(0);
    if (!SvROK(func) || SvTYPE(SvRV(func)) != SVt_PVCV) {
        croak("util::partial: first argument must be a coderef");
    }

    IV idx = g_partial_count++;
    ensure_partial_capacity(idx);

    PartialFunc *pf = &g_partials[idx];
    pf->func = SvREFCNT_inc_simple_NN(func);
    pf->bound_args = newAV();

    /* Store bound arguments */
    IV i;
    for (i = 1; i < items; i++) {
        av_push(pf->bound_args, SvREFCNT_inc_simple_NN(ST(i)));
    }

    CV *wrapper = newXS(NULL, xs_partial_call, __FILE__);
    CvXSUBANY(wrapper).any_iv = idx;

    ST(0) = newRV_noinc((SV*)wrapper);
    XSRETURN(1);
}

static XS(xs_partial_call) {
    dXSARGS;
    IV idx = CvXSUBANY(cv).any_iv;
    PartialFunc *pf = &g_partials[idx];

    IV bound_count = av_len(pf->bound_args) + 1;
    IV total = bound_count + items;

    ENTER;
    SAVETMPS;
    PUSHMARK(SP);

    EXTEND(SP, total);

    /* Push bound args first */
    IV i;
    for (i = 0; i < bound_count; i++) {
        SV **elem = av_fetch(pf->bound_args, i, 0);
        PUSHs(elem ? *elem : &PL_sv_undef);
    }

    /* Push call-time args */
    for (i = 0; i < items; i++) {
        PUSHs(ST(i));
    }
    PUTBACK;

    IV count = call_sv(pf->func, G_SCALAR);

    SPAGAIN;
    SV *result = count > 0 ? POPs : &PL_sv_undef;
    SvREFCNT_inc(result);
    PUTBACK;
    FREETMPS;
    LEAVE;

    ST(0) = result;
    XSRETURN(1);
}

/* ============================================
   Data extraction functions
   ============================================ */

/* pick($hash, @keys) - extract subset of keys */
static XS(xs_pick) {
    dXSARGS;
    if (items < 1) croak("Usage: util::pick(\\%%hash, @keys)");

    SV *href = ST(0);
    if (!SvROK(href) || SvTYPE(SvRV(href)) != SVt_PVHV) {
        croak("util::pick: first argument must be a hashref");
    }

    HV *src = (HV*)SvRV(href);
    HV *dest = newHV();

    IV i;
    for (i = 1; i < items; i++) {
        SV *key = ST(i);
        STRLEN key_len;
        const char *key_pv = SvPV(key, key_len);

        SV **val = hv_fetch(src, key_pv, key_len, 0);
        if (val && SvOK(*val)) {
            hv_store(dest, key_pv, key_len, SvREFCNT_inc(*val), 0);
        }
    }

    ST(0) = newRV_noinc((SV*)dest);
    XSRETURN(1);
}

/* pluck(\@hashes, $field) - extract field from each hash */
static XS(xs_pluck) {
    dXSARGS;
    if (items != 2) croak("Usage: util::pluck(\\@array, $field)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::pluck: first argument must be an arrayref");
    }

    SV *field = ST(1);
    STRLEN field_len;
    const char *field_pv = SvPV(field, field_len);

    AV *src = (AV*)SvRV(aref);
    IV len = av_len(src) + 1;
    AV *dest = newAV();
    av_extend(dest, len - 1);

    IV i;
    for (i = 0; i < len; i++) {
        SV **elem = av_fetch(src, i, 0);
        if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
            HV *hv = (HV*)SvRV(*elem);
            SV **val = hv_fetch(hv, field_pv, field_len, 0);
            if (val && SvOK(*val)) {
                av_push(dest, SvREFCNT_inc(*val));
            } else {
                av_push(dest, &PL_sv_undef);
            }
        } else {
            av_push(dest, &PL_sv_undef);
        }
    }

    ST(0) = newRV_noinc((SV*)dest);
    XSRETURN(1);
}

/* omit($hash, @keys) - exclude subset of keys (inverse of pick) */
static XS(xs_omit) {
    dXSARGS;
    if (items < 1) croak("Usage: util::omit(\\%%hash, @keys)");

    SV *href = ST(0);
    if (!SvROK(href) || SvTYPE(SvRV(href)) != SVt_PVHV) {
        croak("util::omit: first argument must be a hashref");
    }

    HV *src = (HV*)SvRV(href);
    HV *dest = newHV();

    /* Build exclusion set for O(1) lookup */
    HV *exclude = newHV();
    IV i;
    for (i = 1; i < items; i++) {
        SV *key = ST(i);
        STRLEN key_len;
        const char *key_pv = SvPV(key, key_len);
        hv_store(exclude, key_pv, key_len, &PL_sv_yes, 0);
    }

    /* Iterate source, copy non-excluded keys */
    hv_iterinit(src);
    HE *entry;
    while ((entry = hv_iternext(src)) != NULL) {
        SV *key = hv_iterkeysv(entry);
        STRLEN key_len;
        const char *key_pv = SvPV(key, key_len);

        if (!hv_exists(exclude, key_pv, key_len)) {
            SV *val = hv_iterval(src, entry);
            if (SvOK(val)) {
                hv_store(dest, key_pv, key_len, SvREFCNT_inc(val), 0);
            }
        }
    }

    SvREFCNT_dec((SV*)exclude);
    ST(0) = newRV_noinc((SV*)dest);
    XSRETURN(1);
}

/* uniq(@list) - return unique elements (preserves order) */
static XS(xs_uniq) {
    dXSARGS;

    if (items == 0) {
        XSRETURN(0);
    }

    if (items == 1) {
        XSRETURN(1);
    }

    /* For small lists, use simple O(n^2) - faster due to no hash overhead */
    if (items <= 8) {
        IV out = 0;
        IV i, j;
        for (i = 0; i < items; i++) {
            SV *elem = ST(i);
            STRLEN len_i;
            const char *key_i = SvOK(elem) ? SvPV_const(elem, len_i) : "\x00UNDEF\x00";
            if (!SvOK(elem)) len_i = 7;
            
            bool dup = FALSE;
            for (j = 0; j < out; j++) {
                SV *prev = ST(j);
                STRLEN len_j;
                const char *key_j = SvOK(prev) ? SvPV_const(prev, len_j) : "\x00UNDEF\x00";
                if (!SvOK(prev)) len_j = 7;
                
                if (len_i == len_j && memcmp(key_i, key_j, len_i) == 0) {
                    dup = TRUE;
                    break;
                }
            }
            if (!dup) ST(out++) = elem;
        }
        XSRETURN(out);
    }

    HV *seen = newHV();
    IV out = 0;
    hv_ksplit(seen, items);

    IV i;
    for (i = 0; i < items; i++) {
        SV *elem = ST(i);
        STRLEN len;
        const char *key;
        U32 hash;

        key = SvOK(elem) ? SvPV_const(elem, len) : (len = 7, "\x00UNDEF\x00");

        PERL_HASH(hash, key, len);

        if (!hv_common(seen, NULL, key, len, 0, HV_FETCH_ISEXISTS, NULL, hash)) {
            hv_common(seen, NULL, key, len, 0, HV_FETCH_ISSTORE, &PL_sv_yes, hash);
            ST(out++) = elem;
        }
    }

    SvREFCNT_dec_NN((SV*)seen);
    XSRETURN(out);
}

/* partition(\&pred, @list) - split into [matches], [non-matches] */
static XS(xs_partition) {
    dXSARGS;
    if (items < 1) croak("Usage: util::partition(\\&block, @list)");

    SV *block = ST(0);
    if (!SvROK(block) || SvTYPE(SvRV(block)) != SVt_PVCV) {
        croak("util::partition: first argument must be a coderef");
    }

    IV list_len = items - 1;
    
    if (list_len == 0) {
        AV *pass = newAV();
        AV *fail = newAV();
        AV *outer = newAV();
        av_push(outer, newRV_noinc((SV*)pass));
        av_push(outer, newRV_noinc((SV*)fail));
        ST(0) = newRV_noinc((SV*)outer);
        XSRETURN(1);
    }

    AV *pass = newAV();
    AV *fail = newAV();
    av_extend(pass, list_len >> 1);
    av_extend(fail, list_len >> 1);

    SV *orig_defsv = DEFSV;
    
    IV i;
    for (i = 1; i < items; i++) {
        SV *elem = ST(i);

        DEFSV_set(elem);

        PUSHMARK(SP);
        *++SP = elem;
        PUTBACK;

        call_sv(block, G_SCALAR);

        SPAGAIN;
        bool matched = SvTRUE(*SP--);
        PUTBACK;

        if (matched) {
            av_push(pass, SvREFCNT_inc_simple_NN(elem));
        } else {
            av_push(fail, SvREFCNT_inc_simple_NN(elem));
        }
    }
    
    DEFSV_set(orig_defsv);

    AV *outer = newAV();
    av_push(outer, newRV_noinc((SV*)pass));
    av_push(outer, newRV_noinc((SV*)fail));

    ST(0) = newRV_noinc((SV*)outer);
    XSRETURN(1);
}

/* defaults($hash, $defaults) - fill in missing keys from defaults */
static XS(xs_defaults) {
    dXSARGS;
    if (items != 2) croak("Usage: util::defaults(\\%%hash, \\%%defaults)");

    SV *href = ST(0);
    SV *dref = ST(1);

    if (!SvROK(href) || SvTYPE(SvRV(href)) != SVt_PVHV) {
        croak("util::defaults: first argument must be a hashref");
    }
    if (!SvROK(dref) || SvTYPE(SvRV(dref)) != SVt_PVHV) {
        croak("util::defaults: second argument must be a hashref");
    }

    HV *src = (HV*)SvRV(href);
    HV *def = (HV*)SvRV(dref);
    
    /* Pre-size dest hash */
    IV src_keys = HvUSEDKEYS(src);
    IV def_keys = HvUSEDKEYS(def);
    HV *dest = newHV();
    hv_ksplit(dest, src_keys + def_keys);

    /* Copy all from source first */
    hv_iterinit(src);
    HE *entry;
    while ((entry = hv_iternext(src)) != NULL) {
        STRLEN key_len;
        const char *key_pv = HePV(entry, key_len);
        SV *val = HeVAL(entry);
        hv_store(dest, key_pv, key_len, SvREFCNT_inc_simple_NN(val), HeHASH(entry));
    }

    /* Fill in missing from defaults - use pre-computed hash */
    hv_iterinit(def);
    while ((entry = hv_iternext(def)) != NULL) {
        STRLEN key_len;
        const char *key_pv = HePV(entry, key_len);
        U32 hash = HeHASH(entry);

        /* Check if exists and is defined in dest */
        SV **existing = hv_fetch(dest, key_pv, key_len, 0);
        if (!existing || !SvOK(*existing)) {
            SV *val = HeVAL(entry);
            hv_store(dest, key_pv, key_len, SvREFCNT_inc_simple_NN(val), hash);
        }
    }

    ST(0) = newRV_noinc((SV*)dest);
    XSRETURN(1);
}

/* ============================================
   Null coalescing functions
   ============================================ */

/* nvl($x, $default) - return $x if defined, else $default */
static XS(xs_nvl) {
    dXSARGS;
    if (items != 2) croak("Usage: util::nvl($value, $default)");

    SV *val = ST(0);
    if (SvOK(val)) {
        XSRETURN(1);  /* Return first arg */
    }
    ST(0) = ST(1);
    XSRETURN(1);
}

/* coalesce($a, $b, ...) - return first defined value */
static XS(xs_coalesce) {
    dXSARGS;
    if (items < 1) croak("Usage: util::coalesce($val, ...)");

    IV i;
    for (i = 0; i < items; i++) {
        if (SvOK(ST(i))) {
            ST(0) = ST(i);
            XSRETURN(1);
        }
    }
    /* All undefined, return undef */
    ST(0) = &PL_sv_undef;
    XSRETURN(1);
}

/* ============================================
   List functions (first, any, all, none)

   These use MULTICALL for pure Perl subs which is significantly
   faster than call_sv() for repeated invocations.

   For XS subs, we fall back to call_sv().
   ============================================ */

/* Inline CALLRUNOPS - experimental optimization to skip function call overhead.
   Use cautiously - this inlines the runops loop directly. */
#define INLINE_RUNOPS() \
    STMT_START { \
        OP *_inline_op = PL_op; \
        while ((_inline_op = _inline_op->op_ppaddr(aTHX))) ; \
    } STMT_END

/* ============================================
   Specialized array predicates - pure C, no callback
   These are blazing fast because they avoid all Perl callback overhead
   ============================================ */

/* first_gt(\@array, $threshold) or first_gt(\@array, $key, $threshold)
   first element > threshold, pure C
   With key: first hash where hash->{key} > threshold */
static XS(xs_first_gt) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::first_gt(\\@array, $threshold) or first_gt(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::first_gt: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        /* Simple array of scalars */
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) > threshold) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        /* Array of hashes with key */
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) > threshold) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* first_lt(\@array, $threshold) or first_lt(\@array, $key, $threshold)
   first element < threshold, pure C */
static XS(xs_first_lt) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::first_lt(\\@array, $threshold) or first_lt(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::first_lt: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) < threshold) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) < threshold) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* first_eq(\@array, $value) or first_eq(\@array, $key, $value)
   first element == value (numeric), pure C */
static XS(xs_first_eq) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::first_eq(\\@array, $value) or first_eq(\\@array, $key, $value)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::first_eq: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV target = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) == target) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV target = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) == target) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* first_ge(\@array, $threshold) or first_ge(\@array, $key, $threshold)
   first element >= threshold, pure C */
static XS(xs_first_ge) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::first_ge(\\@array, $threshold) or first_ge(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::first_ge: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) >= threshold) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) >= threshold) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* first_le(\@array, $threshold) or first_le(\@array, $key, $threshold)
   first element <= threshold, pure C */
static XS(xs_first_le) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::first_le(\\@array, $threshold) or first_le(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::first_le: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) <= threshold) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) <= threshold) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* first_ne(\@array, $value) or first_ne(\@array, $key, $value)
   first element != value (numeric), pure C */
static XS(xs_first_ne) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::first_ne(\\@array, $value) or first_ne(\\@array, $key, $value)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::first_ne: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV target = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) != target) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV target = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) != target) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* ============================================
   final_* - like first_* but iterates backwards
   ============================================ */

/* final_gt(\@array, $threshold) or final_gt(\@array, $key, $threshold)
   last element > threshold, pure C, backwards iteration */
static XS(xs_final_gt) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::final_gt(\\@array, $threshold) or final_gt(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::final_gt: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) > threshold) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) > threshold) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* final_lt(\@array, $threshold) or final_lt(\@array, $key, $threshold) */
static XS(xs_final_lt) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::final_lt(\\@array, $threshold) or final_lt(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::final_lt: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) < threshold) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) < threshold) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* final_ge(\@array, $threshold) or final_ge(\@array, $key, $threshold) */
static XS(xs_final_ge) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::final_ge(\\@array, $threshold) or final_ge(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::final_ge: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) >= threshold) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) >= threshold) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* final_le(\@array, $threshold) or final_le(\@array, $key, $threshold) */
static XS(xs_final_le) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::final_le(\\@array, $threshold) or final_le(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::final_le: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) <= threshold) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) <= threshold) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* final_eq(\@array, $value) or final_eq(\@array, $key, $value) */
static XS(xs_final_eq) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::final_eq(\\@array, $value) or final_eq(\\@array, $key, $value)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::final_eq: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV target = SvNV(ST(1));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) == target) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV target = SvNV(ST(2));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) == target) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* final_ne(\@array, $value) or final_ne(\@array, $key, $value) */
static XS(xs_final_ne) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::final_ne(\\@array, $value) or final_ne(\\@array, $key, $value)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::final_ne: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV target = SvNV(ST(1));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) != target) {
                ST(0) = *elem;
                XSRETURN(1);
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV target = SvNV(ST(2));
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) != target) {
                    ST(0) = *elem;
                    XSRETURN(1);
                }
            }
        }
    }

    XSRETURN_UNDEF;
}

/* any_gt(\@array, $threshold) or any_gt(\@array, $key, $threshold)
   true if any element > threshold, pure C */
static XS(xs_any_gt) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::any_gt(\\@array, $threshold) or any_gt(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::any_gt: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) > threshold) {
                XSRETURN_YES;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) > threshold) {
                    XSRETURN_YES;
                }
            }
        }
    }

    XSRETURN_NO;
}

/* any_lt(\@array, $threshold) or any_lt(\@array, $key, $threshold) */
static XS(xs_any_lt) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::any_lt(\\@array, $threshold) or any_lt(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::any_lt: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) < threshold) {
                XSRETURN_YES;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) < threshold) {
                    XSRETURN_YES;
                }
            }
        }
    }

    XSRETURN_NO;
}

/* any_ge(\@array, $threshold) or any_ge(\@array, $key, $threshold) */
static XS(xs_any_ge) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::any_ge(\\@array, $threshold) or any_ge(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::any_ge: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) >= threshold) {
                XSRETURN_YES;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) >= threshold) {
                    XSRETURN_YES;
                }
            }
        }
    }

    XSRETURN_NO;
}

/* any_le(\@array, $threshold) or any_le(\@array, $key, $threshold) */
static XS(xs_any_le) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::any_le(\\@array, $threshold) or any_le(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::any_le: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) <= threshold) {
                XSRETURN_YES;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) <= threshold) {
                    XSRETURN_YES;
                }
            }
        }
    }

    XSRETURN_NO;
}

/* any_eq(\@array, $value) or any_eq(\@array, $key, $value) */
static XS(xs_any_eq) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::any_eq(\\@array, $value) or any_eq(\\@array, $key, $value)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::any_eq: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV target = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) == target) {
                XSRETURN_YES;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV target = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) == target) {
                    XSRETURN_YES;
                }
            }
        }
    }

    XSRETURN_NO;
}

/* any_ne(\@array, $value) or any_ne(\@array, $key, $value) */
static XS(xs_any_ne) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::any_ne(\\@array, $value) or any_ne(\\@array, $key, $value)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::any_ne: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV target = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) != target) {
                XSRETURN_YES;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV target = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) != target) {
                    XSRETURN_YES;
                }
            }
        }
    }

    XSRETURN_NO;
}

/* all_gt(\@array, $n) - true if all elements > n, pure C */
/* all_gt(\@array, $threshold) or all_gt(\@array, $key, $threshold)
   true if all elements > threshold, pure C */
static XS(xs_all_gt) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::all_gt(\\@array, $threshold) or all_gt(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::all_gt: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (len == 0) XSRETURN_YES; /* vacuous truth */

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || SvNV(*elem) <= threshold) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || !SvROK(*elem) || SvTYPE(SvRV(*elem)) != SVt_PVHV) {
                XSRETURN_NO;
            }
            HV *hv = (HV *)SvRV(*elem);
            SV **val = hv_fetch(hv, key, strlen(key), 0);
            if (!val || SvNV(*val) <= threshold) {
                XSRETURN_NO;
            }
        }
    }

    XSRETURN_YES;
}

/* all_lt(\@array, $threshold) or all_lt(\@array, $key, $threshold) */
static XS(xs_all_lt) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::all_lt(\\@array, $threshold) or all_lt(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::all_lt: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (len == 0) XSRETURN_YES;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || SvNV(*elem) >= threshold) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || !SvROK(*elem) || SvTYPE(SvRV(*elem)) != SVt_PVHV) {
                XSRETURN_NO;
            }
            HV *hv = (HV *)SvRV(*elem);
            SV **val = hv_fetch(hv, key, strlen(key), 0);
            if (!val || SvNV(*val) >= threshold) {
                XSRETURN_NO;
            }
        }
    }

    XSRETURN_YES;
}

/* all_ge(\@array, $threshold) or all_ge(\@array, $key, $threshold) */
static XS(xs_all_ge) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::all_ge(\\@array, $threshold) or all_ge(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::all_ge: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (len == 0) XSRETURN_YES;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || SvNV(*elem) < threshold) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || !SvROK(*elem) || SvTYPE(SvRV(*elem)) != SVt_PVHV) {
                XSRETURN_NO;
            }
            HV *hv = (HV *)SvRV(*elem);
            SV **val = hv_fetch(hv, key, strlen(key), 0);
            if (!val || SvNV(*val) < threshold) {
                XSRETURN_NO;
            }
        }
    }

    XSRETURN_YES;
}

/* all_le(\@array, $threshold) or all_le(\@array, $key, $threshold) */
static XS(xs_all_le) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::all_le(\\@array, $threshold) or all_le(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::all_le: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (len == 0) XSRETURN_YES;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || SvNV(*elem) > threshold) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || !SvROK(*elem) || SvTYPE(SvRV(*elem)) != SVt_PVHV) {
                XSRETURN_NO;
            }
            HV *hv = (HV *)SvRV(*elem);
            SV **val = hv_fetch(hv, key, strlen(key), 0);
            if (!val || SvNV(*val) > threshold) {
                XSRETURN_NO;
            }
        }
    }

    XSRETURN_YES;
}

/* all_eq(\@array, $value) or all_eq(\@array, $key, $value) */
static XS(xs_all_eq) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::all_eq(\\@array, $value) or all_eq(\\@array, $key, $value)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::all_eq: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (len == 0) XSRETURN_YES;

    if (items == 2) {
        NV target = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || SvNV(*elem) != target) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV target = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || !SvROK(*elem) || SvTYPE(SvRV(*elem)) != SVt_PVHV) {
                XSRETURN_NO;
            }
            HV *hv = (HV *)SvRV(*elem);
            SV **val = hv_fetch(hv, key, strlen(key), 0);
            if (!val || SvNV(*val) != target) {
                XSRETURN_NO;
            }
        }
    }

    XSRETURN_YES;
}

/* all_ne(\@array, $value) or all_ne(\@array, $key, $value) */
static XS(xs_all_ne) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::all_ne(\\@array, $value) or all_ne(\\@array, $key, $value)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::all_ne: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (len == 0) XSRETURN_YES;

    if (items == 2) {
        NV target = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || SvNV(*elem) == target) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV target = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem || !SvROK(*elem) || SvTYPE(SvRV(*elem)) != SVt_PVHV) {
                XSRETURN_NO;
            }
            HV *hv = (HV *)SvRV(*elem);
            SV **val = hv_fetch(hv, key, strlen(key), 0);
            if (!val || SvNV(*val) == target) {
                XSRETURN_NO;
            }
        }
    }

    XSRETURN_YES;
}

/* none_gt(\@array, $threshold) or none_gt(\@array, $key, $threshold)
   true if no element > threshold, pure C */
static XS(xs_none_gt) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::none_gt(\\@array, $threshold) or none_gt(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::none_gt: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) > threshold) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) > threshold) {
                    XSRETURN_NO;
                }
            }
        }
    }

    XSRETURN_YES;
}

/* none_lt(\@array, $threshold) or none_lt(\@array, $key, $threshold) */
static XS(xs_none_lt) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::none_lt(\\@array, $threshold) or none_lt(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::none_lt: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) < threshold) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) < threshold) {
                    XSRETURN_NO;
                }
            }
        }
    }

    XSRETURN_YES;
}

/* none_ge(\@array, $threshold) or none_ge(\@array, $key, $threshold) */
static XS(xs_none_ge) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::none_ge(\\@array, $threshold) or none_ge(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::none_ge: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) >= threshold) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) >= threshold) {
                    XSRETURN_NO;
                }
            }
        }
    }

    XSRETURN_YES;
}

/* none_le(\@array, $threshold) or none_le(\@array, $key, $threshold) */
static XS(xs_none_le) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::none_le(\\@array, $threshold) or none_le(\\@array, $key, $threshold)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::none_le: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV threshold = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) <= threshold) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV threshold = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) <= threshold) {
                    XSRETURN_NO;
                }
            }
        }
    }

    XSRETURN_YES;
}

/* none_eq(\@array, $value) or none_eq(\@array, $key, $value) */
static XS(xs_none_eq) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::none_eq(\\@array, $value) or none_eq(\\@array, $key, $value)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::none_eq: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV target = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) == target) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV target = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) == target) {
                    XSRETURN_NO;
                }
            }
        }
    }

    XSRETURN_YES;
}

/* none_ne(\@array, $value) or none_ne(\@array, $key, $value) */
static XS(xs_none_ne) {
    dXSARGS;
    if (items < 2 || items > 3) croak("Usage: util::none_ne(\\@array, $value) or none_ne(\\@array, $key, $value)");

    SV *aref = ST(0);
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::none_ne: first argument must be an arrayref");
    }

    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (items == 2) {
        NV target = SvNV(ST(1));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvNV(*elem) != target) {
                XSRETURN_NO;
            }
        }
    } else {
        char *key = SvPV_nolen(ST(1));
        NV target = SvNV(ST(2));
        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVHV) {
                HV *hv = (HV *)SvRV(*elem);
                SV **val = hv_fetch(hv, key, strlen(key), 0);
                if (val && SvNV(*val) != target) {
                    XSRETURN_NO;
                }
            }
        }
    }

    XSRETURN_YES;
}

/* firstr(\&block, \@array) - first with arrayref, no stack flattening */
static XS(xs_firstr) {
    dXSARGS;
    if (items != 2) croak("Usage: util::firstr(\\&block, \\@array)");

    SV *block = ST(0);
    SV *aref = ST(1);

    if (!SvROK(block) || SvTYPE(SvRV(block)) != SVt_PVCV) {
        croak("util::firstr: first argument must be a coderef");
    }
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::firstr: second argument must be an arrayref");
    }

    CV *block_cv = (CV *)SvRV(block);
    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (len == 0) {
        XSRETURN_UNDEF;
    }

#ifdef dMULTICALL
    if (!CvISXSUB(block_cv)) {
        dMULTICALL;
        I32 gimme = G_SCALAR;

        SAVESPTR(GvSV(PL_defgv));
        PUSH_MULTICALL(block_cv);

        for (i = 0; i < len; i++) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem) continue;

            SV *def_sv = GvSV(PL_defgv) = *elem;
            SvTEMP_off(def_sv);

            MULTICALL;

            if (SvTRUE(*PL_stack_sp)) {
                POP_MULTICALL;
                ST(0) = *elem;
                XSRETURN(1);
            }
        }

        POP_MULTICALL;
        XSRETURN_UNDEF;
    }
#endif

    /* Fallback for XS subs */
    for (i = 0; i < len; i++) {
        SV **elem = av_fetch(av, i, 0);
        if (!elem) continue;

        dSP;
        GvSV(PL_defgv) = *elem;

        PUSHMARK(SP);
        call_sv((SV*)block_cv, G_SCALAR);

        if (SvTRUE(*PL_stack_sp)) {
            ST(0) = *elem;
            XSRETURN(1);
        }
    }

    XSRETURN_UNDEF;
}

/* final(\&block, \@array) - last element where block returns true (backwards iteration) */
static XS(xs_final) {
    dXSARGS;
    if (items != 2) croak("Usage: util::final(\\&block, \\@array)");

    SV *block = ST(0);
    SV *aref = ST(1);

    if (!SvROK(block) || SvTYPE(SvRV(block)) != SVt_PVCV) {
        croak("util::final: first argument must be a coderef");
    }
    if (!SvROK(aref) || SvTYPE(SvRV(aref)) != SVt_PVAV) {
        croak("util::final: second argument must be an arrayref");
    }

    CV *block_cv = (CV *)SvRV(block);
    AV *av = (AV *)SvRV(aref);
    SSize_t len = av_len(av) + 1;
    SSize_t i;

    if (len == 0) {
        XSRETURN_UNDEF;
    }

#ifdef dMULTICALL
    if (!CvISXSUB(block_cv)) {
        dMULTICALL;
        I32 gimme = G_SCALAR;

        SAVESPTR(GvSV(PL_defgv));
        PUSH_MULTICALL(block_cv);

        /* Iterate backwards for speed */
        for (i = len - 1; i >= 0; i--) {
            SV **elem = av_fetch(av, i, 0);
            if (!elem) continue;

            SV *def_sv = GvSV(PL_defgv) = *elem;
            SvTEMP_off(def_sv);

            MULTICALL;

            if (SvTRUE(*PL_stack_sp)) {
                POP_MULTICALL;
                ST(0) = *elem;
                XSRETURN(1);
            }
        }

        POP_MULTICALL;
        XSRETURN_UNDEF;
    }
#endif

    /* Fallback for XS subs - backwards */
    for (i = len - 1; i >= 0; i--) {
        SV **elem = av_fetch(av, i, 0);
        if (!elem) continue;

        dSP;
        GvSV(PL_defgv) = *elem;

        PUSHMARK(SP);
        call_sv((SV*)block_cv, G_SCALAR);

        if (SvTRUE(*PL_stack_sp)) {
            ST(0) = *elem;
            XSRETURN(1);
        }
    }

    XSRETURN_UNDEF;
}

/* first { block } @list - return first element where block returns true */
static XS(xs_first) {
    dXSARGS;
    if (items < 1) croak("Usage: util::first(\\&block, @list)");

    SV *block = ST(0);
    if (!SvROK(block) || SvTYPE(SvRV(block)) != SVt_PVCV) {
        croak("util::first: first argument must be a coderef");
    }

    CV *block_cv = (CV *)SvRV(block);
    /* Store args from stack base before any stack manipulation */
    SV **args = &PL_stack_base[ax];
    IV index;

    /* Empty list - return undef */
    if (items <= 1) {
        XSRETURN_UNDEF;
    }

    /* Use MULTICALL for pure Perl subs - much faster than call_sv */
#ifdef dMULTICALL
    if (!CvISXSUB(block_cv)) {
        dMULTICALL;
        I32 gimme = G_SCALAR;

        SAVESPTR(GvSV(PL_defgv));
        PUSH_MULTICALL(block_cv);

        for (index = 1; index < items; index++) {
            SV *def_sv = GvSV(PL_defgv) = args[index];
            SvTEMP_off(def_sv);

            MULTICALL;

            if (SvTRUE(*PL_stack_sp)) {
                POP_MULTICALL;
                ST(0) = ST(index);
                XSRETURN(1);
            }
        }

        POP_MULTICALL;
        XSRETURN_UNDEF;
    }
#endif

    /* Fallback for XS subs */
    for (index = 1; index < items; index++) {
        dSP;
        GvSV(PL_defgv) = args[index];

        PUSHMARK(SP);
        call_sv((SV*)block_cv, G_SCALAR);

        if (SvTRUE(*PL_stack_sp)) {
            ST(0) = ST(index);
            XSRETURN(1);
        }
    }

    XSRETURN_UNDEF;
}

/* any { block } @list - return true if any element matches */
static XS(xs_any) {
    dXSARGS;
    if (items < 1) croak("Usage: util::any(\\&block, @list)");

    SV *block = ST(0);
    if (!SvROK(block) || SvTYPE(SvRV(block)) != SVt_PVCV) {
        croak("util::any: first argument must be a coderef");
    }

    CV *block_cv = (CV *)SvRV(block);
    SV **args = &PL_stack_base[ax];
    IV index;

    /* Empty list returns false */
    if (items <= 1) {
        XSRETURN_NO;
    }

#ifdef dMULTICALL
    if (!CvISXSUB(block_cv)) {
        dMULTICALL;
        I32 gimme = G_SCALAR;

        SAVESPTR(GvSV(PL_defgv));
        PUSH_MULTICALL(block_cv);

        for (index = 1; index < items; index++) {
            SV *def_sv = GvSV(PL_defgv) = args[index];
            SvTEMP_off(def_sv);

            MULTICALL;

            if (SvTRUE(*PL_stack_sp)) {
                POP_MULTICALL;
                XSRETURN_YES;
            }
        }

        POP_MULTICALL;
        XSRETURN_NO;
    }
#endif

    for (index = 1; index < items; index++) {
        dSP;
        GvSV(PL_defgv) = args[index];

        PUSHMARK(SP);
        call_sv((SV*)block_cv, G_SCALAR);

        if (SvTRUE(*PL_stack_sp)) {
            XSRETURN_YES;
        }
    }

    XSRETURN_NO;
}

/* all { block } @list - return true if all elements match */
static XS(xs_all) {
    dXSARGS;
    if (items < 1) croak("Usage: util::all(\\&block, @list)");

    SV *block = ST(0);
    if (!SvROK(block) || SvTYPE(SvRV(block)) != SVt_PVCV) {
        croak("util::all: first argument must be a coderef");
    }

    CV *block_cv = (CV *)SvRV(block);
    SV **args = &PL_stack_base[ax];
    IV index;

    /* Empty list returns true (vacuous truth) */
    if (items <= 1) {
        XSRETURN_YES;
    }

#ifdef dMULTICALL
    if (!CvISXSUB(block_cv)) {
        dMULTICALL;
        I32 gimme = G_SCALAR;

        SAVESPTR(GvSV(PL_defgv));
        PUSH_MULTICALL(block_cv);

        for (index = 1; index < items; index++) {
            SV *def_sv = GvSV(PL_defgv) = args[index];
            SvTEMP_off(def_sv);

            MULTICALL;

            if (!SvTRUE(*PL_stack_sp)) {
                POP_MULTICALL;
                XSRETURN_NO;
            }
        }

        POP_MULTICALL;
        XSRETURN_YES;
    }
#endif

    for (index = 1; index < items; index++) {
        dSP;
        GvSV(PL_defgv) = args[index];

        PUSHMARK(SP);
        call_sv((SV*)block_cv, G_SCALAR);

        if (!SvTRUE(*PL_stack_sp)) {
            XSRETURN_NO;
        }
    }

    XSRETURN_YES;
}

/* none { block } @list - return true if no elements match */
static XS(xs_none) {
    dXSARGS;
    if (items < 1) croak("Usage: util::none(\\&block, @list)");

    SV *block = ST(0);
    if (!SvROK(block) || SvTYPE(SvRV(block)) != SVt_PVCV) {
        croak("util::none: first argument must be a coderef");
    }

    CV *block_cv = (CV *)SvRV(block);
    SV **args = &PL_stack_base[ax];
    IV index;

    /* Empty list returns true (no elements match = vacuous truth) */
    if (items <= 1) {
        XSRETURN_YES;
    }

#ifdef dMULTICALL
    if (!CvISXSUB(block_cv)) {
        dMULTICALL;
        I32 gimme = G_SCALAR;

        SAVESPTR(GvSV(PL_defgv));
        PUSH_MULTICALL(block_cv);

        for (index = 1; index < items; index++) {
            SV *def_sv = GvSV(PL_defgv) = args[index];
            SvTEMP_off(def_sv);

            MULTICALL;

            if (SvTRUE(*PL_stack_sp)) {
                POP_MULTICALL;
                XSRETURN_NO;
            }
        }

        POP_MULTICALL;
        XSRETURN_YES;
    }
#endif

    for (index = 1; index < items; index++) {
        dSP;
        GvSV(PL_defgv) = args[index];

        PUSHMARK(SP);
        call_sv((SV*)block_cv, G_SCALAR);

        if (SvTRUE(*PL_stack_sp)) {
            XSRETURN_NO;
        }
    }

    XSRETURN_YES;
}

/* ============================================
   Experimental: Inlined MULTICALL versions for benchmarking

   These versions inline the runops loop to skip the CALLRUNOPS
   function call overhead. For testing only.
   ============================================ */

/* first_inline - experimental version with inlined runops loop */
static XS(xs_first_inline) {
    dXSARGS;
    if (items < 1) croak("Usage: util::first_inline(\\&block, @list)");

    SV *block = ST(0);
    if (!SvROK(block) || SvTYPE(SvRV(block)) != SVt_PVCV) {
        croak("util::first_inline: first argument must be a coderef");
    }

    CV *the_cv = (CV *)SvRV(block);

    if (items == 1) {
        XSRETURN_UNDEF;
    }

    /* Only works with pure Perl subs */
    if (CvISXSUB(the_cv)) {
        croak("util::first_inline: only works with pure Perl subs");
    }

    SV **args = &ST(1);
    IV num_args = items - 1;
    IV i;

    /* Manually set up the MULTICALL context */
    PERL_CONTEXT *cx;
    bool multicall_oldcatch;
    OP *multicall_cop;
    I32 gimme = G_SCALAR;

    PADLIST *padlist = CvPADLIST(the_cv);

    multicall_oldcatch = CATCH_GET;
    CATCH_SET(TRUE);
    PUSHSTACKi(PERLSI_MULTICALL);

    cx = cx_pushblock((CXt_SUB|CXp_MULTICALL), (U8)gimme,
                      PL_stack_sp, PL_savestack_ix);
    cx_pushsub(cx, the_cv, NULL, 0);

    SAVEOP();
    CvDEPTH(the_cv)++;
    if (CvDEPTH(the_cv) >= 2)
        Perl_pad_push(aTHX_ padlist, CvDEPTH(the_cv));
    PAD_SET_CUR_NOSAVE(padlist, CvDEPTH(the_cv));

    multicall_cop = CvSTART(the_cv);

    /* Save and setup $_ */
    SAVESPTR(GvSV(PL_defgv));

    for (i = 0; i < num_args; i++) {
        SV *elem = args[i];
        GvSV(PL_defgv) = elem;
        SvTEMP_off(elem);

        /* Inlined MULTICALL - skip CALLRUNOPS function call */
        PL_op = multicall_cop;
        {
            OP *op = PL_op;
            while ((PL_op = op = op->op_ppaddr(aTHX))) ;
        }

        if (SvTRUE(*PL_stack_sp)) {
            /* Found it - cleanup and return */
            cx = CX_CUR();
            CX_LEAVE_SCOPE(cx);
            cx_popsub_common(cx);
            cx_popblock(cx);
            CX_POP(cx);
            POPSTACK;
            CATCH_SET(multicall_oldcatch);
            SPAGAIN;

            ST(0) = elem;
            XSRETURN(1);
        }
    }

    /* Cleanup - POP_MULTICALL equivalent */
    cx = CX_CUR();
    CX_LEAVE_SCOPE(cx);
    cx_popsub_common(cx);
    cx_popblock(cx);
    CX_POP(cx);
    POPSTACK;
    CATCH_SET(multicall_oldcatch);
    SPAGAIN;

    XSRETURN_UNDEF;
}


/* ============================================
   Type predicate XS fallbacks
   ============================================ */

static XS(xs_is_ref) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_ref($value)");
    ST(0) = SvROK(ST(0)) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

static XS(xs_is_array) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_array($value)");
    SV *sv = ST(0);
    ST(0) = (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVAV) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

static XS(xs_is_hash) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_hash($value)");
    SV *sv = ST(0);
    ST(0) = (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVHV) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

static XS(xs_is_code) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_code($value)");
    SV *sv = ST(0);
    ST(0) = (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVCV) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

static XS(xs_is_defined) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_defined($value)");
    ST(0) = SvOK(ST(0)) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

/* ============================================
   String predicate XS fallbacks
   ============================================ */

static XS(xs_is_empty) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_empty($value)");
    SV *sv = ST(0);
    if (!SvOK(sv)) {
        ST(0) = &PL_sv_yes;
    } else {
        STRLEN len;
        SvPV(sv, len);
        ST(0) = len == 0 ? &PL_sv_yes : &PL_sv_no;
    }
    XSRETURN(1);
}

static XS(xs_starts_with) {
    dXSARGS;
    if (items != 2) croak("Usage: util::starts_with($string, $prefix)");

    SV *str_sv = ST(0);
    SV *prefix_sv = ST(1);

    if (!SvOK(str_sv) || !SvOK(prefix_sv)) {
        ST(0) = &PL_sv_no;
        XSRETURN(1);
    }

    STRLEN str_len, prefix_len;
    const char *str = SvPV(str_sv, str_len);
    const char *prefix = SvPV(prefix_sv, prefix_len);

    if (prefix_len > str_len) {
        ST(0) = &PL_sv_no;
    } else if (prefix_len == 0) {
        ST(0) = &PL_sv_yes;
    } else {
        ST(0) = memcmp(str, prefix, prefix_len) == 0 ? &PL_sv_yes : &PL_sv_no;
    }
    XSRETURN(1);
}

static XS(xs_ends_with) {
    dXSARGS;
    if (items != 2) croak("Usage: util::ends_with($string, $suffix)");

    SV *str_sv = ST(0);
    SV *suffix_sv = ST(1);

    if (!SvOK(str_sv) || !SvOK(suffix_sv)) {
        ST(0) = &PL_sv_no;
        XSRETURN(1);
    }

    STRLEN str_len, suffix_len;
    const char *str = SvPV(str_sv, str_len);
    const char *suffix = SvPV(suffix_sv, suffix_len);

    if (suffix_len > str_len) {
        ST(0) = &PL_sv_no;
    } else if (suffix_len == 0) {
        ST(0) = &PL_sv_yes;
    } else {
        const char *str_end = str + str_len - suffix_len;
        ST(0) = memcmp(str_end, suffix, suffix_len) == 0 ? &PL_sv_yes : &PL_sv_no;
    }
    XSRETURN(1);
}

/* count: count occurrences of substring using memmem */
static XS(xs_count) {
    dXSARGS;
    if (items != 2) croak("Usage: util::count($string, $substring)");

    SV *str_sv = ST(0);
    SV *needle_sv = ST(1);

    if (!SvOK(str_sv) || !SvOK(needle_sv)) {
        ST(0) = sv_2mortal(newSViv(0));
        XSRETURN(1);
    }

    STRLEN str_len, needle_len;
    const char *str = SvPV_const(str_sv, str_len);
    const char *needle = SvPV_const(needle_sv, needle_len);

    if (needle_len == 0 || needle_len > str_len) {
        ST(0) = sv_2mortal(newSViv(0));
        XSRETURN(1);
    }

    IV count = 0;
    const char *p = str;
    const char *end = str + str_len;
    STRLEN remaining = str_len;

    while (remaining >= needle_len) {
        const char *found = (const char *)util_memmem(p, remaining, needle, needle_len);
        if (!found) break;
        count++;
        /* Move past the match (non-overlapping) */
        p = found + needle_len;
        remaining = end - p;
    }

    ST(0) = sv_2mortal(newSViv(count));
    XSRETURN(1);
}

/* replace_all: replace all occurrences of old with new using memmem */
static XS(xs_replace_all) {
    dXSARGS;
    if (items != 3) croak("Usage: util::replace_all($string, $old, $new)");

    SV *str_sv = ST(0);
    SV *old_sv = ST(1);
    SV *new_sv = ST(2);

    /* Handle undef - return undef */
    if (!SvOK(str_sv)) {
        ST(0) = &PL_sv_undef;
        XSRETURN(1);
    }

    STRLEN str_len, old_len, new_len;
    const char *str = SvPV_const(str_sv, str_len);
    const char *old = SvPV_const(old_sv, old_len);
    const char *replacement = SvPV_const(new_sv, new_len);

    /* Empty search string or not found - return original */
    if (old_len == 0 || old_len > str_len) {
        ST(0) = sv_2mortal(newSVpvn(str, str_len));
        XSRETURN(1);
    }

    /* First pass: count occurrences to pre-size buffer */
    IV count = 0;
    const char *p = str;
    const char *end = str + str_len;
    STRLEN remaining = str_len;

    while (remaining >= old_len) {
        const char *found = (const char *)util_memmem(p, remaining, old, old_len);
        if (!found) break;
        count++;
        p = found + old_len;
        remaining = end - p;
    }

    if (count == 0) {
        /* No matches - return copy of original */
        ST(0) = sv_2mortal(newSVpvn(str, str_len));
        XSRETURN(1);
    }

    /* Calculate result size and allocate */
    STRLEN result_len = str_len + count * (new_len - old_len);
    SV *result = sv_2mortal(newSV(result_len + 1));
    SvPOK_on(result);
    char *out = SvPVX(result);
    char *out_ptr = out;

    /* Second pass: build result */
    p = str;
    remaining = str_len;

    while (remaining >= old_len) {
        const char *found = (const char *)util_memmem(p, remaining, old, old_len);
        if (!found) break;

        /* Copy text before match */
        STRLEN before_len = found - p;
        if (before_len > 0) {
            memcpy(out_ptr, p, before_len);
            out_ptr += before_len;
        }

        /* Copy replacement */
        if (new_len > 0) {
            memcpy(out_ptr, replacement, new_len);
            out_ptr += new_len;
        }

        p = found + old_len;
        remaining = end - p;
    }

    /* Copy remaining text after last match */
    if (remaining > 0) {
        memcpy(out_ptr, p, remaining);
        out_ptr += remaining;
    }

    *out_ptr = '\0';
    SvCUR_set(result, out_ptr - out);

    ST(0) = result;
    XSRETURN(1);
}

/* before: get text before first occurrence of delimiter */
static XS(xs_before) {
    dXSARGS;
    if (items != 2) croak("Usage: util::before($string, $delimiter)");

    SV *str_sv = ST(0);
    SV *delim_sv = ST(1);

    if (!SvOK(str_sv)) {
        ST(0) = &PL_sv_undef;
        XSRETURN(1);
    }

    STRLEN str_len, delim_len;
    const char *str = SvPV_const(str_sv, str_len);
    const char *delim = SvPV_const(delim_sv, delim_len);

    if (delim_len == 0 || delim_len > str_len) {
        ST(0) = sv_2mortal(newSVpvn(str, str_len));
        XSRETURN(1);
    }

    const char *found = (const char *)util_memmem(str, str_len, delim, delim_len);
    if (found) {
        ST(0) = sv_2mortal(newSVpvn(str, found - str));
    } else {
        ST(0) = sv_2mortal(newSVpvn(str, str_len));
    }
    XSRETURN(1);
}

/* after: get text after first occurrence of delimiter */
static XS(xs_after) {
    dXSARGS;
    if (items != 2) croak("Usage: util::after($string, $delimiter)");

    SV *str_sv = ST(0);
    SV *delim_sv = ST(1);

    if (!SvOK(str_sv)) {
        ST(0) = &PL_sv_undef;
        XSRETURN(1);
    }

    STRLEN str_len, delim_len;
    const char *str = SvPV_const(str_sv, str_len);
    const char *delim = SvPV_const(delim_sv, delim_len);

    if (delim_len == 0 || delim_len > str_len) {
        ST(0) = sv_2mortal(newSVpvn("", 0));
        XSRETURN(1);
    }

    const char *found = (const char *)util_memmem(str, str_len, delim, delim_len);
    if (found) {
        const char *after_delim = found + delim_len;
        ST(0) = sv_2mortal(newSVpvn(after_delim, str + str_len - after_delim));
    } else {
        ST(0) = sv_2mortal(newSVpvn("", 0));
    }
    XSRETURN(1);
}

/* ============================================
   Boolean/Truthiness XS fallbacks
   ============================================ */

static XS(xs_is_true) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_true($value)");
    ST(0) = SvTRUE(ST(0)) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

static XS(xs_is_false) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_false($value)");
    ST(0) = SvTRUE(ST(0)) ? &PL_sv_no : &PL_sv_yes;
    XSRETURN(1);
}

static XS(xs_bool) {
    dXSARGS;
    if (items != 1) croak("Usage: util::bool($value)");
    ST(0) = SvTRUE(ST(0)) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

/* ============================================
   Extended type predicate XS fallbacks
   ============================================ */

static XS(xs_is_num) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_num($value)");
    SV *sv = ST(0);
    ST(0) = (SvNIOK(sv) || looks_like_number(sv)) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

static XS(xs_is_int) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_int($value)");
    SV *sv = ST(0);
    if (SvIOK(sv)) {
        ST(0) = &PL_sv_yes;
    } else if (SvNOK(sv)) {
        NV nv = SvNV(sv);
        ST(0) = (nv == (NV)(IV)nv) ? &PL_sv_yes : &PL_sv_no;
    } else if (looks_like_number(sv)) {
        STRLEN len;
        const char *pv = SvPV(sv, len);
        bool has_dot = FALSE;
        STRLEN i;
        for (i = 0; i < len; i++) {
            if (pv[i] == '.' || pv[i] == 'e' || pv[i] == 'E') {
                has_dot = TRUE;
                break;
            }
        }
        if (has_dot) {
            NV nv = SvNV(sv);
            ST(0) = (nv == (NV)(IV)nv) ? &PL_sv_yes : &PL_sv_no;
        } else {
            ST(0) = &PL_sv_yes;
        }
    } else {
        ST(0) = &PL_sv_no;
    }
    XSRETURN(1);
}

static XS(xs_is_blessed) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_blessed($value)");
    ST(0) = sv_isobject(ST(0)) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

static XS(xs_is_scalar_ref) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_scalar_ref($value)");
    SV *sv = ST(0);
    if (SvROK(sv)) {
        SV *rv = SvRV(sv);
        svtype type = SvTYPE(rv);
        ST(0) = (type < SVt_PVAV) ? &PL_sv_yes : &PL_sv_no;
    } else {
        ST(0) = &PL_sv_no;
    }
    XSRETURN(1);
}

static XS(xs_is_regex) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_regex($value)");
    ST(0) = SvRXOK(ST(0)) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

static XS(xs_is_glob) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_glob($value)");
    ST(0) = (SvTYPE(ST(0)) == SVt_PVGV) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

static XS(xs_is_string) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_string($value)");
    SV *sv = ST(0);
    ST(0) = (SvOK(sv) && !SvROK(sv)) ? &PL_sv_yes : &PL_sv_no;
    XSRETURN(1);
}

/* ============================================
   Numeric predicate XS fallbacks
   ============================================ */

static XS(xs_is_positive) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_positive($value)");
    SV *sv = ST(0);
    if (SvNIOK(sv) || looks_like_number(sv)) {
        NV nv = SvNV(sv);
        ST(0) = (nv > 0) ? &PL_sv_yes : &PL_sv_no;
    } else {
        ST(0) = &PL_sv_no;
    }
    XSRETURN(1);
}

static XS(xs_is_negative) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_negative($value)");
    SV *sv = ST(0);
    if (SvNIOK(sv) || looks_like_number(sv)) {
        NV nv = SvNV(sv);
        ST(0) = (nv < 0) ? &PL_sv_yes : &PL_sv_no;
    } else {
        ST(0) = &PL_sv_no;
    }
    XSRETURN(1);
}

static XS(xs_is_zero) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_zero($value)");
    SV *sv = ST(0);
    if (SvNIOK(sv) || looks_like_number(sv)) {
        NV nv = SvNV(sv);
        ST(0) = (nv == 0) ? &PL_sv_yes : &PL_sv_no;
    } else {
        ST(0) = &PL_sv_no;
    }
    XSRETURN(1);
}

/* ============================================
   Numeric utility XS fallbacks
   ============================================ */

static XS(xs_is_even) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_even($value)");
    SV *sv = ST(0);
    if (SvIOK(sv)) {
        ST(0) = (SvIVX(sv) & 1) == 0 ? &PL_sv_yes : &PL_sv_no;
    } else if (SvNIOK(sv)) {
        NV nv = SvNV(sv);
        if (nv == (NV)(IV)nv) {
            ST(0) = ((IV)nv & 1) == 0 ? &PL_sv_yes : &PL_sv_no;
        } else {
            ST(0) = &PL_sv_no;
        }
    } else if (looks_like_number(sv)) {
        ST(0) = (SvIV(sv) & 1) == 0 ? &PL_sv_yes : &PL_sv_no;
    } else {
        ST(0) = &PL_sv_no;
    }
    XSRETURN(1);
}

static XS(xs_is_odd) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_odd($value)");
    SV *sv = ST(0);
    if (SvIOK(sv)) {
        ST(0) = (SvIVX(sv) & 1) == 1 ? &PL_sv_yes : &PL_sv_no;
    } else if (SvNIOK(sv)) {
        NV nv = SvNV(sv);
        if (nv == (NV)(IV)nv) {
            ST(0) = ((IV)nv & 1) == 1 ? &PL_sv_yes : &PL_sv_no;
        } else {
            ST(0) = &PL_sv_no;
        }
    } else if (looks_like_number(sv)) {
        ST(0) = (SvIV(sv) & 1) == 1 ? &PL_sv_yes : &PL_sv_no;
    } else {
        ST(0) = &PL_sv_no;
    }
    XSRETURN(1);
}

static XS(xs_is_between) {
    dXSARGS;
    if (items != 3) croak("Usage: util::is_between($value, $min, $max)");
    SV *val_sv = ST(0);
    SV *min_sv = ST(1);
    SV *max_sv = ST(2);

    if (SvNIOK(val_sv) || looks_like_number(val_sv)) {
        NV val = SvNV(val_sv);
        NV min = SvNV(min_sv);
        NV max = SvNV(max_sv);
        ST(0) = (val >= min && val <= max) ? &PL_sv_yes : &PL_sv_no;
    } else {
        ST(0) = &PL_sv_no;
    }
    XSRETURN(1);
}

/* ============================================
   Collection XS fallbacks
   ============================================ */

static XS(xs_is_empty_array) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_empty_array($arrayref)");
    SV *sv = ST(0);
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVAV) {
        AV *av = (AV*)SvRV(sv);
        ST(0) = AvFILL(av) < 0 ? &PL_sv_yes : &PL_sv_no;
    } else {
        ST(0) = &PL_sv_no;
    }
    XSRETURN(1);
}

static XS(xs_is_empty_hash) {
    dXSARGS;
    if (items != 1) croak("Usage: util::is_empty_hash($hashref)");
    SV *sv = ST(0);
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVHV) {
        HV *hv = (HV*)SvRV(sv);
        ST(0) = HvKEYS(hv) == 0 ? &PL_sv_yes : &PL_sv_no;
    } else {
        ST(0) = &PL_sv_no;
    }
    XSRETURN(1);
}

static XS(xs_array_len) {
    dXSARGS;
    if (items != 1) croak("Usage: util::array_len($arrayref)");
    SV *sv = ST(0);
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVAV) {
        AV *av = (AV*)SvRV(sv);
        ST(0) = sv_2mortal(newSViv(AvFILL(av) + 1));
    } else {
        ST(0) = &PL_sv_undef;
    }
    XSRETURN(1);
}

static XS(xs_hash_size) {
    dXSARGS;
    if (items != 1) croak("Usage: util::hash_size($hashref)");
    SV *sv = ST(0);
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVHV) {
        HV *hv = (HV*)SvRV(sv);
        ST(0) = sv_2mortal(newSViv(HvKEYS(hv)));
    } else {
        ST(0) = &PL_sv_undef;
    }
    XSRETURN(1);
}

static XS(xs_array_first) {
    dXSARGS;
    if (items != 1) croak("Usage: util::array_first($arrayref)");
    SV *sv = ST(0);
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVAV) {
        AV *av = (AV*)SvRV(sv);
        if (AvFILL(av) >= 0) {
            SV **elem = av_fetch(av, 0, 0);
            ST(0) = elem ? *elem : &PL_sv_undef;
        } else {
            ST(0) = &PL_sv_undef;
        }
    } else {
        ST(0) = &PL_sv_undef;
    }
    XSRETURN(1);
}

static XS(xs_array_last) {
    dXSARGS;
    if (items != 1) croak("Usage: util::array_last($arrayref)");
    SV *sv = ST(0);
    if (SvROK(sv) && SvTYPE(SvRV(sv)) == SVt_PVAV) {
        AV *av = (AV*)SvRV(sv);
        IV last_idx = AvFILL(av);
        if (last_idx >= 0) {
            SV **elem = av_fetch(av, last_idx, 0);
            ST(0) = elem ? *elem : &PL_sv_undef;
        } else {
            ST(0) = &PL_sv_undef;
        }
    } else {
        ST(0) = &PL_sv_undef;
    }
    XSRETURN(1);
}

/* ============================================
   String manipulation XS fallbacks
   ============================================ */

static XS(xs_trim) {
    dXSARGS;
    if (items != 1) croak("Usage: util::trim($string)");

    SV *sv = ST(0);
    if (!SvOK(sv)) {
        ST(0) = &PL_sv_undef;
        XSRETURN(1);
    }

    STRLEN len;
    const char *str = SvPV(sv, len);
    const char *start = str;
    const char *end = str + len;

    /* Skip leading whitespace */
    while (start < end && isSPACE(*start)) {
        start++;
    }

    /* Skip trailing whitespace */
    while (end > start && isSPACE(*(end - 1))) {
        end--;
    }

    ST(0) = sv_2mortal(newSVpvn(start, end - start));
    XSRETURN(1);
}

static XS(xs_ltrim) {
    dXSARGS;
    if (items != 1) croak("Usage: util::ltrim($string)");

    SV *sv = ST(0);
    if (!SvOK(sv)) {
        ST(0) = &PL_sv_undef;
        XSRETURN(1);
    }

    STRLEN len;
    const char *str = SvPV(sv, len);
    const char *start = str;
    const char *end = str + len;

    while (start < end && isSPACE(*start)) {
        start++;
    }

    ST(0) = sv_2mortal(newSVpvn(start, end - start));
    XSRETURN(1);
}

static XS(xs_rtrim) {
    dXSARGS;
    if (items != 1) croak("Usage: util::rtrim($string)");

    SV *sv = ST(0);
    if (!SvOK(sv)) {
        ST(0) = &PL_sv_undef;
        XSRETURN(1);
    }

    STRLEN len;
    const char *str = SvPV(sv, len);
    const char *end = str + len;

    while (end > str && isSPACE(*(end - 1))) {
        end--;
    }

    ST(0) = sv_2mortal(newSVpvn(str, end - str));
    XSRETURN(1);
}

/* ============================================
   Conditional XS fallbacks
   ============================================ */

static XS(xs_maybe) {
    dXSARGS;
    if (items != 2) croak("Usage: util::maybe($value, $then)");

    SV *val = ST(0);
    if (SvOK(val)) {
        ST(0) = ST(1);
    } else {
        ST(0) = &PL_sv_undef;
    }
    XSRETURN(1);
}

/* ============================================
   Numeric XS fallbacks
   ============================================ */

static XS(xs_sign) {
    dXSARGS;
    if (items != 1) croak("Usage: util::sign($number)");

    SV *sv = ST(0);
    if (!SvNIOK(sv) && !looks_like_number(sv)) {
        ST(0) = &PL_sv_undef;
        XSRETURN(1);
    }

    NV nv = SvNV(sv);
    if (nv > 0) {
        ST(0) = sv_2mortal(newSViv(1));
    } else if (nv < 0) {
        ST(0) = sv_2mortal(newSViv(-1));
    } else {
        ST(0) = sv_2mortal(newSViv(0));
    }
    XSRETURN(1);
}

static XS(xs_min2) {
    dXSARGS;
    if (items != 2) croak("Usage: util::min2($a, $b)");

    NV a = SvNV(ST(0));
    NV b = SvNV(ST(1));

    ST(0) = a <= b ? ST(0) : ST(1);
    XSRETURN(1);
}

static XS(xs_max2) {
    dXSARGS;
    if (items != 2) croak("Usage: util::max2($a, $b)");

    NV a = SvNV(ST(0));
    NV b = SvNV(ST(1));

    ST(0) = a >= b ? ST(0) : ST(1);
    XSRETURN(1);
}

/* ============================================
   Import function
   ============================================ */

static char* get_caller(pTHX) {
    return HvNAME((HV*)CopSTASH(PL_curcop));
}

static XS(xs_import) {
    dXSARGS;
    char *pkg = get_caller(aTHX);
    IV i;

    for (i = 1; i < items; i++) {
        char *name = SvPV_nolen(ST(i));
        char full[512];
        CV *cv = NULL;

        snprintf(full, sizeof(full), "%s::%s", pkg, name);

        if (strcmp(name, "memo") == 0) {
            cv = newXS(full, xs_memo, __FILE__);
        } else if (strcmp(name, "pipeline") == 0) {
            cv = newXS(full, xs_pipe, __FILE__);
        } else if (strcmp(name, "compose") == 0) {
            cv = newXS(full, xs_compose, __FILE__);
        } else if (strcmp(name, "lazy") == 0) {
            cv = newXS(full, xs_lazy, __FILE__);
        } else if (strcmp(name, "force") == 0) {
            cv = newXS(full, xs_force, __FILE__);
        } else if (strcmp(name, "dig") == 0) {
            cv = newXS(full, xs_dig, __FILE__);
        } else if (strcmp(name, "clamp") == 0) {
            cv = newXS(full, xs_clamp, __FILE__);
        } else if (strcmp(name, "tap") == 0) {
            cv = newXS(full, xs_tap, __FILE__);
        } else if (strcmp(name, "identity") == 0) {
            cv = newXS(full, xs_identity, __FILE__);
            cv_set_call_checker(cv, identity_call_checker, (SV*)cv);
        } else if (strcmp(name, "always") == 0) {
            cv = newXS(full, xs_always, __FILE__);
        } else if (strcmp(name, "noop") == 0) {
            cv = newXS(full, xs_noop, __FILE__);
            cv_set_call_checker(cv, noop_call_checker, (SV*)cv);
        } else if (strcmp(name, "stub_true") == 0) {
            cv = newXS(full, xs_stub_true, __FILE__);
        } else if (strcmp(name, "stub_false") == 0) {
            cv = newXS(full, xs_stub_false, __FILE__);
        } else if (strcmp(name, "stub_array") == 0) {
            cv = newXS(full, xs_stub_array, __FILE__);
        } else if (strcmp(name, "stub_hash") == 0) {
            cv = newXS(full, xs_stub_hash, __FILE__);
        } else if (strcmp(name, "stub_string") == 0) {
            cv = newXS(full, xs_stub_string, __FILE__);
        } else if (strcmp(name, "stub_zero") == 0) {
            cv = newXS(full, xs_stub_zero, __FILE__);
        } else if (strcmp(name, "nvl") == 0) {
            cv = newXS(full, xs_nvl, __FILE__);
        } else if (strcmp(name, "coalesce") == 0) {
            cv = newXS(full, xs_coalesce, __FILE__);
        } else if (strcmp(name, "first") == 0) {
            cv = newXS(full, xs_first, __FILE__);
        } else if (strcmp(name, "firstr") == 0) {
            cv = newXS(full, xs_firstr, __FILE__);
        } else if (strcmp(name, "first_gt") == 0) {
            cv = newXS(full, xs_first_gt, __FILE__);
        } else if (strcmp(name, "first_lt") == 0) {
            cv = newXS(full, xs_first_lt, __FILE__);
        } else if (strcmp(name, "first_ge") == 0) {
            cv = newXS(full, xs_first_ge, __FILE__);
        } else if (strcmp(name, "first_le") == 0) {
            cv = newXS(full, xs_first_le, __FILE__);
        } else if (strcmp(name, "first_eq") == 0) {
            cv = newXS(full, xs_first_eq, __FILE__);
        } else if (strcmp(name, "first_ne") == 0) {
            cv = newXS(full, xs_first_ne, __FILE__);
        } else if (strcmp(name, "final") == 0) {
            cv = newXS(full, xs_final, __FILE__);
        } else if (strcmp(name, "final_gt") == 0) {
            cv = newXS(full, xs_final_gt, __FILE__);
        } else if (strcmp(name, "final_lt") == 0) {
            cv = newXS(full, xs_final_lt, __FILE__);
        } else if (strcmp(name, "final_ge") == 0) {
            cv = newXS(full, xs_final_ge, __FILE__);
        } else if (strcmp(name, "final_le") == 0) {
            cv = newXS(full, xs_final_le, __FILE__);
        } else if (strcmp(name, "final_eq") == 0) {
            cv = newXS(full, xs_final_eq, __FILE__);
        } else if (strcmp(name, "final_ne") == 0) {
            cv = newXS(full, xs_final_ne, __FILE__);
        } else if (strcmp(name, "any_gt") == 0) {
            cv = newXS(full, xs_any_gt, __FILE__);
        } else if (strcmp(name, "any_lt") == 0) {
            cv = newXS(full, xs_any_lt, __FILE__);
        } else if (strcmp(name, "any_ge") == 0) {
            cv = newXS(full, xs_any_ge, __FILE__);
        } else if (strcmp(name, "any_le") == 0) {
            cv = newXS(full, xs_any_le, __FILE__);
        } else if (strcmp(name, "any_eq") == 0) {
            cv = newXS(full, xs_any_eq, __FILE__);
        } else if (strcmp(name, "any_ne") == 0) {
            cv = newXS(full, xs_any_ne, __FILE__);
        } else if (strcmp(name, "all_gt") == 0) {
            cv = newXS(full, xs_all_gt, __FILE__);
        } else if (strcmp(name, "all_lt") == 0) {
            cv = newXS(full, xs_all_lt, __FILE__);
        } else if (strcmp(name, "all_ge") == 0) {
            cv = newXS(full, xs_all_ge, __FILE__);
        } else if (strcmp(name, "all_le") == 0) {
            cv = newXS(full, xs_all_le, __FILE__);
        } else if (strcmp(name, "all_eq") == 0) {
            cv = newXS(full, xs_all_eq, __FILE__);
        } else if (strcmp(name, "all_ne") == 0) {
            cv = newXS(full, xs_all_ne, __FILE__);
        } else if (strcmp(name, "none_gt") == 0) {
            cv = newXS(full, xs_none_gt, __FILE__);
        } else if (strcmp(name, "none_lt") == 0) {
            cv = newXS(full, xs_none_lt, __FILE__);
        } else if (strcmp(name, "none_ge") == 0) {
            cv = newXS(full, xs_none_ge, __FILE__);
        } else if (strcmp(name, "none_le") == 0) {
            cv = newXS(full, xs_none_le, __FILE__);
        } else if (strcmp(name, "none_eq") == 0) {
            cv = newXS(full, xs_none_eq, __FILE__);
        } else if (strcmp(name, "none_ne") == 0) {
            cv = newXS(full, xs_none_ne, __FILE__);
        } else if (strcmp(name, "first_inline") == 0) {
            cv = newXS(full, xs_first_inline, __FILE__);
        } else if (strcmp(name, "any") == 0) {
            cv = newXS(full, xs_any, __FILE__);
        } else if (strcmp(name, "all") == 0) {
            cv = newXS(full, xs_all, __FILE__);
        } else if (strcmp(name, "none") == 0) {
            cv = newXS(full, xs_none, __FILE__);
        } else if (strcmp(name, "negate") == 0) {
            cv = newXS(full, xs_negate, __FILE__);
        } else if (strcmp(name, "once") == 0) {
            cv = newXS(full, xs_once, __FILE__);
        } else if (strcmp(name, "partial") == 0) {
            cv = newXS(full, xs_partial, __FILE__);
        } else if (strcmp(name, "pick") == 0) {
            cv = newXS(full, xs_pick, __FILE__);
        } else if (strcmp(name, "pluck") == 0) {
            cv = newXS(full, xs_pluck, __FILE__);
        } else if (strcmp(name, "omit") == 0) {
            cv = newXS(full, xs_omit, __FILE__);
        } else if (strcmp(name, "uniq") == 0) {
            cv = newXS(full, xs_uniq, __FILE__);
        } else if (strcmp(name, "partition") == 0) {
            cv = newXS(full, xs_partition, __FILE__);
        } else if (strcmp(name, "defaults") == 0) {
            cv = newXS(full, xs_defaults, __FILE__);
        } else if (strcmp(name, "count") == 0) {
            cv = newXS(full, xs_count, __FILE__);
        } else if (strcmp(name, "replace_all") == 0) {
            cv = newXS(full, xs_replace_all, __FILE__);
        } else if (strcmp(name, "is_ref") == 0) {
            cv = newXS(full, xs_is_ref, __FILE__);
            cv_set_call_checker(cv, is_ref_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_array") == 0) {
            cv = newXS(full, xs_is_array, __FILE__);
            cv_set_call_checker(cv, is_array_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_hash") == 0) {
            cv = newXS(full, xs_is_hash, __FILE__);
            cv_set_call_checker(cv, is_hash_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_code") == 0) {
            cv = newXS(full, xs_is_code, __FILE__);
            cv_set_call_checker(cv, is_code_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_defined") == 0) {
            cv = newXS(full, xs_is_defined, __FILE__);
            cv_set_call_checker(cv, is_defined_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_empty") == 0) {
            cv = newXS(full, xs_is_empty, __FILE__);
            cv_set_call_checker(cv, is_empty_call_checker, (SV*)cv);
        } else if (strcmp(name, "starts_with") == 0) {
            cv = newXS(full, xs_starts_with, __FILE__);
            cv_set_call_checker(cv, starts_with_call_checker, (SV*)cv);
        } else if (strcmp(name, "ends_with") == 0) {
            cv = newXS(full, xs_ends_with, __FILE__);
            cv_set_call_checker(cv, ends_with_call_checker, (SV*)cv);
        /* Boolean/Truthiness predicates */
        } else if (strcmp(name, "is_true") == 0) {
            cv = newXS(full, xs_is_true, __FILE__);
            cv_set_call_checker(cv, is_true_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_false") == 0) {
            cv = newXS(full, xs_is_false, __FILE__);
            cv_set_call_checker(cv, is_false_call_checker, (SV*)cv);
        } else if (strcmp(name, "bool") == 0) {
            cv = newXS(full, xs_bool, __FILE__);
            cv_set_call_checker(cv, bool_call_checker, (SV*)cv);
        /* Extended type predicates */
        } else if (strcmp(name, "is_num") == 0) {
            cv = newXS(full, xs_is_num, __FILE__);
            cv_set_call_checker(cv, is_num_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_int") == 0) {
            cv = newXS(full, xs_is_int, __FILE__);
            cv_set_call_checker(cv, is_int_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_blessed") == 0) {
            cv = newXS(full, xs_is_blessed, __FILE__);
            cv_set_call_checker(cv, is_blessed_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_scalar_ref") == 0) {
            cv = newXS(full, xs_is_scalar_ref, __FILE__);
            cv_set_call_checker(cv, is_scalar_ref_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_regex") == 0) {
            cv = newXS(full, xs_is_regex, __FILE__);
            cv_set_call_checker(cv, is_regex_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_glob") == 0) {
            cv = newXS(full, xs_is_glob, __FILE__);
            cv_set_call_checker(cv, is_glob_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_string") == 0) {
            cv = newXS(full, xs_is_string, __FILE__);
            cv_set_call_checker(cv, is_string_call_checker, (SV*)cv);
        /* Numeric predicates */
        } else if (strcmp(name, "is_positive") == 0) {
            cv = newXS(full, xs_is_positive, __FILE__);
            cv_set_call_checker(cv, is_positive_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_negative") == 0) {
            cv = newXS(full, xs_is_negative, __FILE__);
            cv_set_call_checker(cv, is_negative_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_zero") == 0) {
            cv = newXS(full, xs_is_zero, __FILE__);
            cv_set_call_checker(cv, is_zero_call_checker, (SV*)cv);
        /* Numeric utility ops */
        } else if (strcmp(name, "is_even") == 0) {
            cv = newXS(full, xs_is_even, __FILE__);
            cv_set_call_checker(cv, is_even_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_odd") == 0) {
            cv = newXS(full, xs_is_odd, __FILE__);
            cv_set_call_checker(cv, is_odd_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_between") == 0) {
            cv = newXS(full, xs_is_between, __FILE__);
            cv_set_call_checker(cv, is_between_call_checker, (SV*)cv);
        /* Collection ops */
        } else if (strcmp(name, "is_empty_array") == 0) {
            cv = newXS(full, xs_is_empty_array, __FILE__);
            cv_set_call_checker(cv, is_empty_array_call_checker, (SV*)cv);
        } else if (strcmp(name, "is_empty_hash") == 0) {
            cv = newXS(full, xs_is_empty_hash, __FILE__);
            cv_set_call_checker(cv, is_empty_hash_call_checker, (SV*)cv);
        } else if (strcmp(name, "array_len") == 0) {
            cv = newXS(full, xs_array_len, __FILE__);
            cv_set_call_checker(cv, array_len_call_checker, (SV*)cv);
        } else if (strcmp(name, "hash_size") == 0) {
            cv = newXS(full, xs_hash_size, __FILE__);
            cv_set_call_checker(cv, hash_size_call_checker, (SV*)cv);
        } else if (strcmp(name, "array_first") == 0) {
            cv = newXS(full, xs_array_first, __FILE__);
            cv_set_call_checker(cv, array_first_call_checker, (SV*)cv);
        } else if (strcmp(name, "array_last") == 0) {
            cv = newXS(full, xs_array_last, __FILE__);
            cv_set_call_checker(cv, array_last_call_checker, (SV*)cv);
        /* String manipulation ops */
        } else if (strcmp(name, "trim") == 0) {
            cv = newXS(full, xs_trim, __FILE__);
            cv_set_call_checker(cv, trim_call_checker, (SV*)cv);
        } else if (strcmp(name, "ltrim") == 0) {
            cv = newXS(full, xs_ltrim, __FILE__);
            cv_set_call_checker(cv, ltrim_call_checker, (SV*)cv);
        } else if (strcmp(name, "rtrim") == 0) {
            cv = newXS(full, xs_rtrim, __FILE__);
            cv_set_call_checker(cv, rtrim_call_checker, (SV*)cv);
        /* Conditional ops */
        } else if (strcmp(name, "maybe") == 0) {
            cv = newXS(full, xs_maybe, __FILE__);
            cv_set_call_checker(cv, maybe_call_checker, (SV*)cv);
        /* Numeric ops */
        } else if (strcmp(name, "sign") == 0) {
            cv = newXS(full, xs_sign, __FILE__);
            cv_set_call_checker(cv, sign_call_checker, (SV*)cv);
        } else if (strcmp(name, "min2") == 0) {
            cv = newXS(full, xs_min2, __FILE__);
            cv_set_call_checker(cv, min2_call_checker, (SV*)cv);
        } else if (strcmp(name, "max2") == 0) {
            cv = newXS(full, xs_max2, __FILE__);
            cv_set_call_checker(cv, max2_call_checker, (SV*)cv);
        } else {
            croak("util: unknown export '%s'", name);
        }
        PERL_UNUSED_VAR(cv);
    }

    XSRETURN_EMPTY;
}

/* ============================================
   Boot
   ============================================ */

XS_EXTERNAL(boot_util) {
    dXSBOOTARGSXSAPIVERCHK;
    PERL_UNUSED_VAR(items);

    /* Register custom ops */
    XopENTRY_set(&identity_xop, xop_name, "identity");
    XopENTRY_set(&identity_xop, xop_desc, "identity passthrough");
    Perl_custom_op_register(aTHX_ pp_identity, &identity_xop);

    XopENTRY_set(&always_xop, xop_name, "always");
    XopENTRY_set(&always_xop, xop_desc, "always return stored value");
    Perl_custom_op_register(aTHX_ pp_always, &always_xop);

    XopENTRY_set(&clamp_xop, xop_name, "clamp");
    XopENTRY_set(&clamp_xop, xop_desc, "clamp value between min and max");
    Perl_custom_op_register(aTHX_ pp_clamp, &clamp_xop);

    /* Register type predicate custom ops */
    XopENTRY_set(&is_ref_xop, xop_name, "is_ref");
    XopENTRY_set(&is_ref_xop, xop_desc, "check if value is a reference");
    Perl_custom_op_register(aTHX_ pp_is_ref, &is_ref_xop);

    XopENTRY_set(&is_array_xop, xop_name, "is_array");
    XopENTRY_set(&is_array_xop, xop_desc, "check if value is an arrayref");
    Perl_custom_op_register(aTHX_ pp_is_array, &is_array_xop);

    XopENTRY_set(&is_hash_xop, xop_name, "is_hash");
    XopENTRY_set(&is_hash_xop, xop_desc, "check if value is a hashref");
    Perl_custom_op_register(aTHX_ pp_is_hash, &is_hash_xop);

    XopENTRY_set(&is_code_xop, xop_name, "is_code");
    XopENTRY_set(&is_code_xop, xop_desc, "check if value is a coderef");
    Perl_custom_op_register(aTHX_ pp_is_code, &is_code_xop);

    XopENTRY_set(&is_defined_xop, xop_name, "is_defined");
    XopENTRY_set(&is_defined_xop, xop_desc, "check if value is defined");
    Perl_custom_op_register(aTHX_ pp_is_defined, &is_defined_xop);

    /* Register string predicate custom ops */
    XopENTRY_set(&is_empty_xop, xop_name, "is_empty");
    XopENTRY_set(&is_empty_xop, xop_desc, "check if string is empty");
    Perl_custom_op_register(aTHX_ pp_is_empty, &is_empty_xop);

    XopENTRY_set(&starts_with_xop, xop_name, "starts_with");
    XopENTRY_set(&starts_with_xop, xop_desc, "check if string starts with prefix");
    Perl_custom_op_register(aTHX_ pp_starts_with, &starts_with_xop);

    XopENTRY_set(&ends_with_xop, xop_name, "ends_with");
    XopENTRY_set(&ends_with_xop, xop_desc, "check if string ends with suffix");
    Perl_custom_op_register(aTHX_ pp_ends_with, &ends_with_xop);

    /* Register boolean/truthiness custom ops */
    XopENTRY_set(&is_true_xop, xop_name, "is_true");
    XopENTRY_set(&is_true_xop, xop_desc, "check if value is truthy");
    Perl_custom_op_register(aTHX_ pp_is_true, &is_true_xop);

    XopENTRY_set(&is_false_xop, xop_name, "is_false");
    XopENTRY_set(&is_false_xop, xop_desc, "check if value is falsy");
    Perl_custom_op_register(aTHX_ pp_is_false, &is_false_xop);

    XopENTRY_set(&bool_xop, xop_name, "bool");
    XopENTRY_set(&bool_xop, xop_desc, "normalize to boolean");
    Perl_custom_op_register(aTHX_ pp_bool, &bool_xop);

    /* Register extended type predicate custom ops */
    XopENTRY_set(&is_num_xop, xop_name, "is_num");
    XopENTRY_set(&is_num_xop, xop_desc, "check if value is numeric");
    Perl_custom_op_register(aTHX_ pp_is_num, &is_num_xop);

    XopENTRY_set(&is_int_xop, xop_name, "is_int");
    XopENTRY_set(&is_int_xop, xop_desc, "check if value is integer");
    Perl_custom_op_register(aTHX_ pp_is_int, &is_int_xop);

    XopENTRY_set(&is_blessed_xop, xop_name, "is_blessed");
    XopENTRY_set(&is_blessed_xop, xop_desc, "check if value is blessed");
    Perl_custom_op_register(aTHX_ pp_is_blessed, &is_blessed_xop);

    XopENTRY_set(&is_scalar_ref_xop, xop_name, "is_scalar_ref");
    XopENTRY_set(&is_scalar_ref_xop, xop_desc, "check if value is scalar reference");
    Perl_custom_op_register(aTHX_ pp_is_scalar_ref, &is_scalar_ref_xop);

    XopENTRY_set(&is_regex_xop, xop_name, "is_regex");
    XopENTRY_set(&is_regex_xop, xop_desc, "check if value is compiled regex");
    Perl_custom_op_register(aTHX_ pp_is_regex, &is_regex_xop);

    XopENTRY_set(&is_glob_xop, xop_name, "is_glob");
    XopENTRY_set(&is_glob_xop, xop_desc, "check if value is glob");
    Perl_custom_op_register(aTHX_ pp_is_glob, &is_glob_xop);

    XopENTRY_set(&is_string_xop, xop_name, "is_string");
    XopENTRY_set(&is_string_xop, xop_desc, "check if value is plain scalar");
    Perl_custom_op_register(aTHX_ pp_is_string, &is_string_xop);

    /* Register numeric predicate custom ops */
    XopENTRY_set(&is_positive_xop, xop_name, "is_positive");
    XopENTRY_set(&is_positive_xop, xop_desc, "check if value is positive");
    Perl_custom_op_register(aTHX_ pp_is_positive, &is_positive_xop);

    XopENTRY_set(&is_negative_xop, xop_name, "is_negative");
    XopENTRY_set(&is_negative_xop, xop_desc, "check if value is negative");
    Perl_custom_op_register(aTHX_ pp_is_negative, &is_negative_xop);

    XopENTRY_set(&is_zero_xop, xop_name, "is_zero");
    XopENTRY_set(&is_zero_xop, xop_desc, "check if value is zero");
    Perl_custom_op_register(aTHX_ pp_is_zero, &is_zero_xop);

    /* Register numeric utility custom ops */
    XopENTRY_set(&is_even_xop, xop_name, "is_even");
    XopENTRY_set(&is_even_xop, xop_desc, "check if integer is even");
    Perl_custom_op_register(aTHX_ pp_is_even, &is_even_xop);

    XopENTRY_set(&is_odd_xop, xop_name, "is_odd");
    XopENTRY_set(&is_odd_xop, xop_desc, "check if integer is odd");
    Perl_custom_op_register(aTHX_ pp_is_odd, &is_odd_xop);

    XopENTRY_set(&is_between_xop, xop_name, "is_between");
    XopENTRY_set(&is_between_xop, xop_desc, "check if value is between min and max");
    Perl_custom_op_register(aTHX_ pp_is_between, &is_between_xop);

    /* Register collection custom ops */
    XopENTRY_set(&is_empty_array_xop, xop_name, "is_empty_array");
    XopENTRY_set(&is_empty_array_xop, xop_desc, "check if arrayref is empty");
    Perl_custom_op_register(aTHX_ pp_is_empty_array, &is_empty_array_xop);

    XopENTRY_set(&is_empty_hash_xop, xop_name, "is_empty_hash");
    XopENTRY_set(&is_empty_hash_xop, xop_desc, "check if hashref is empty");
    Perl_custom_op_register(aTHX_ pp_is_empty_hash, &is_empty_hash_xop);

    XopENTRY_set(&array_len_xop, xop_name, "array_len");
    XopENTRY_set(&array_len_xop, xop_desc, "get array length");
    Perl_custom_op_register(aTHX_ pp_array_len, &array_len_xop);

    XopENTRY_set(&hash_size_xop, xop_name, "hash_size");
    XopENTRY_set(&hash_size_xop, xop_desc, "get hash key count");
    Perl_custom_op_register(aTHX_ pp_hash_size, &hash_size_xop);

    XopENTRY_set(&array_first_xop, xop_name, "array_first");
    XopENTRY_set(&array_first_xop, xop_desc, "get first array element");
    Perl_custom_op_register(aTHX_ pp_array_first, &array_first_xop);

    XopENTRY_set(&array_last_xop, xop_name, "array_last");
    XopENTRY_set(&array_last_xop, xop_desc, "get last array element");
    Perl_custom_op_register(aTHX_ pp_array_last, &array_last_xop);

    /* Register string manipulation custom ops */
    XopENTRY_set(&trim_xop, xop_name, "trim");
    XopENTRY_set(&trim_xop, xop_desc, "trim whitespace from string");
    Perl_custom_op_register(aTHX_ pp_trim, &trim_xop);

    XopENTRY_set(&ltrim_xop, xop_name, "ltrim");
    XopENTRY_set(&ltrim_xop, xop_desc, "trim leading whitespace");
    Perl_custom_op_register(aTHX_ pp_ltrim, &ltrim_xop);

    XopENTRY_set(&rtrim_xop, xop_name, "rtrim");
    XopENTRY_set(&rtrim_xop, xop_desc, "trim trailing whitespace");
    Perl_custom_op_register(aTHX_ pp_rtrim, &rtrim_xop);

    /* Register conditional custom ops */
    XopENTRY_set(&maybe_xop, xop_name, "maybe");
    XopENTRY_set(&maybe_xop, xop_desc, "return value if defined");
    Perl_custom_op_register(aTHX_ pp_maybe, &maybe_xop);

    /* Register numeric custom ops */
    XopENTRY_set(&sign_xop, xop_name, "sign");
    XopENTRY_set(&sign_xop, xop_desc, "return sign of number");
    Perl_custom_op_register(aTHX_ pp_sign, &sign_xop);

    XopENTRY_set(&min2_xop, xop_name, "min2");
    XopENTRY_set(&min2_xop, xop_desc, "return smaller of two values");
    Perl_custom_op_register(aTHX_ pp_min2, &min2_xop);

    XopENTRY_set(&max2_xop, xop_name, "max2");
    XopENTRY_set(&max2_xop, xop_desc, "return larger of two values");
    Perl_custom_op_register(aTHX_ pp_max2, &max2_xop);

    /* Initialize memo storage */
    g_memo_size = 16;
    Newxz(g_memos, g_memo_size, MemoizedFunc);

    /* Initialize lazy storage */
    g_lazy_size = 16;
    Newxz(g_lazies, g_lazy_size, LazyValue);

    /* Initialize always storage */
    g_always_size = 16;
    Newxz(g_always_values, g_always_size, SV*);

    /* Initialize once storage */
    g_once_size = 16;
    Newxz(g_onces, g_once_size, OnceFunc);

    /* Initialize partial storage */
    g_partial_size = 16;
    Newxz(g_partials, g_partial_size, PartialFunc);

    /* Export functions */
    newXS("util::import", xs_import, __FILE__);
    newXS("util::memo", xs_memo, __FILE__);
    newXS("util::pipeline", xs_pipe, __FILE__);
    newXS("util::compose", xs_compose, __FILE__);
    newXS("util::lazy", xs_lazy, __FILE__);
    newXS("util::force", xs_force, __FILE__);
    newXS("util::dig", xs_dig, __FILE__);
    
    {
        CV *cv = newXS("util::clamp", xs_clamp, __FILE__);
        cv_set_call_checker(cv, clamp_call_checker, (SV*)cv);
    }
    
    newXS("util::tap", xs_tap, __FILE__);

    {
        CV *cv = newXS("util::identity", xs_identity, __FILE__);
        cv_set_call_checker(cv, identity_call_checker, (SV*)cv);
    }

    newXS("util::always", xs_always, __FILE__);
    {
        CV *cv = newXS("util::noop", xs_noop, __FILE__);
        cv_set_call_checker(cv, noop_call_checker, (SV*)cv);
    }
    newXS("util::stub_true", xs_stub_true, __FILE__);
    newXS("util::stub_false", xs_stub_false, __FILE__);
    newXS("util::stub_array", xs_stub_array, __FILE__);
    newXS("util::stub_hash", xs_stub_hash, __FILE__);
    newXS("util::stub_string", xs_stub_string, __FILE__);
    newXS("util::stub_zero", xs_stub_zero, __FILE__);
    newXS("util::nvl", xs_nvl, __FILE__);
    newXS("util::coalesce", xs_coalesce, __FILE__);

    /* List functions */
    newXS("util::first", xs_first, __FILE__);
    newXS("util::firstr", xs_firstr, __FILE__);
    newXS("util::any", xs_any, __FILE__);
    newXS("util::all", xs_all, __FILE__);
    newXS("util::none", xs_none, __FILE__);
    newXS("util::first_inline", xs_first_inline, __FILE__); /* experimental */

    /* Specialized array predicates - pure C, no callback */
    newXS("util::first_gt", xs_first_gt, __FILE__);
    newXS("util::first_lt", xs_first_lt, __FILE__);
    newXS("util::first_ge", xs_first_ge, __FILE__);
    newXS("util::first_le", xs_first_le, __FILE__);
    newXS("util::first_eq", xs_first_eq, __FILE__);
    newXS("util::first_ne", xs_first_ne, __FILE__);
    newXS("util::final", xs_final, __FILE__);
    newXS("util::final_gt", xs_final_gt, __FILE__);
    newXS("util::final_lt", xs_final_lt, __FILE__);
    newXS("util::final_ge", xs_final_ge, __FILE__);
    newXS("util::final_le", xs_final_le, __FILE__);
    newXS("util::final_eq", xs_final_eq, __FILE__);
    newXS("util::final_ne", xs_final_ne, __FILE__);
    newXS("util::any_gt", xs_any_gt, __FILE__);
    newXS("util::any_lt", xs_any_lt, __FILE__);
    newXS("util::any_ge", xs_any_ge, __FILE__);
    newXS("util::any_le", xs_any_le, __FILE__);
    newXS("util::any_eq", xs_any_eq, __FILE__);
    newXS("util::any_ne", xs_any_ne, __FILE__);
    newXS("util::all_gt", xs_all_gt, __FILE__);
    newXS("util::all_lt", xs_all_lt, __FILE__);
    newXS("util::all_ge", xs_all_ge, __FILE__);
    newXS("util::all_le", xs_all_le, __FILE__);
    newXS("util::all_eq", xs_all_eq, __FILE__);
    newXS("util::all_ne", xs_all_ne, __FILE__);
    newXS("util::none_gt", xs_none_gt, __FILE__);
    newXS("util::none_lt", xs_none_lt, __FILE__);
    newXS("util::none_ge", xs_none_ge, __FILE__);
    newXS("util::none_le", xs_none_le, __FILE__);
    newXS("util::none_eq", xs_none_eq, __FILE__);
    newXS("util::none_ne", xs_none_ne, __FILE__);

    /* Functional combinators */
    newXS("util::negate", xs_negate, __FILE__);
    newXS("util::once", xs_once, __FILE__);
    newXS("util::partial", xs_partial, __FILE__);

    /* Data extraction */
    newXS("util::pick", xs_pick, __FILE__);
    newXS("util::pluck", xs_pluck, __FILE__);
    newXS("util::omit", xs_omit, __FILE__);
    newXS("util::uniq", xs_uniq, __FILE__);
    newXS("util::partition", xs_partition, __FILE__);
    newXS("util::defaults", xs_defaults, __FILE__);

    /* Type predicates with call checkers */
    {
        CV *cv = newXS("util::is_ref", xs_is_ref, __FILE__);
        cv_set_call_checker(cv, is_ref_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_array", xs_is_array, __FILE__);
        cv_set_call_checker(cv, is_array_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_hash", xs_is_hash, __FILE__);
        cv_set_call_checker(cv, is_hash_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_code", xs_is_code, __FILE__);
        cv_set_call_checker(cv, is_code_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_defined", xs_is_defined, __FILE__);
        cv_set_call_checker(cv, is_defined_call_checker, (SV*)cv);
    }

    /* String predicates with call checkers */
    {
        CV *cv = newXS("util::is_empty", xs_is_empty, __FILE__);
        cv_set_call_checker(cv, is_empty_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::starts_with", xs_starts_with, __FILE__);
        cv_set_call_checker(cv, starts_with_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::ends_with", xs_ends_with, __FILE__);
        cv_set_call_checker(cv, ends_with_call_checker, (SV*)cv);
    }
    newXS("util::count", xs_count, __FILE__);
    newXS("util::replace_all", xs_replace_all, __FILE__);

    /* Boolean/Truthiness predicates with call checkers */
    {
        CV *cv = newXS("util::is_true", xs_is_true, __FILE__);
        cv_set_call_checker(cv, is_true_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_false", xs_is_false, __FILE__);
        cv_set_call_checker(cv, is_false_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::bool", xs_bool, __FILE__);
        cv_set_call_checker(cv, bool_call_checker, (SV*)cv);
    }

    /* Extended type predicates with call checkers */
    {
        CV *cv = newXS("util::is_num", xs_is_num, __FILE__);
        cv_set_call_checker(cv, is_num_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_int", xs_is_int, __FILE__);
        cv_set_call_checker(cv, is_int_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_blessed", xs_is_blessed, __FILE__);
        cv_set_call_checker(cv, is_blessed_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_scalar_ref", xs_is_scalar_ref, __FILE__);
        cv_set_call_checker(cv, is_scalar_ref_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_regex", xs_is_regex, __FILE__);
        cv_set_call_checker(cv, is_regex_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_glob", xs_is_glob, __FILE__);
        cv_set_call_checker(cv, is_glob_call_checker, (SV*)cv);
    }

    /* Numeric predicates with call checkers */
    {
        CV *cv = newXS("util::is_positive", xs_is_positive, __FILE__);
        cv_set_call_checker(cv, is_positive_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_negative", xs_is_negative, __FILE__);
        cv_set_call_checker(cv, is_negative_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_zero", xs_is_zero, __FILE__);
        cv_set_call_checker(cv, is_zero_call_checker, (SV*)cv);
    }

    /* Numeric utility ops with call checkers */
    {
        CV *cv = newXS("util::is_even", xs_is_even, __FILE__);
        cv_set_call_checker(cv, is_even_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_odd", xs_is_odd, __FILE__);
        cv_set_call_checker(cv, is_odd_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_between", xs_is_between, __FILE__);
        cv_set_call_checker(cv, is_between_call_checker, (SV*)cv);
    }

    /* Collection ops with call checkers */
    {
        CV *cv = newXS("util::is_empty_array", xs_is_empty_array, __FILE__);
        cv_set_call_checker(cv, is_empty_array_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::is_empty_hash", xs_is_empty_hash, __FILE__);
        cv_set_call_checker(cv, is_empty_hash_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::array_len", xs_array_len, __FILE__);
        cv_set_call_checker(cv, array_len_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::hash_size", xs_hash_size, __FILE__);
        cv_set_call_checker(cv, hash_size_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::array_first", xs_array_first, __FILE__);
        cv_set_call_checker(cv, array_first_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::array_last", xs_array_last, __FILE__);
        cv_set_call_checker(cv, array_last_call_checker, (SV*)cv);
    }

    /* String manipulation ops with call checkers */
    {
        CV *cv = newXS("util::trim", xs_trim, __FILE__);
        cv_set_call_checker(cv, trim_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::ltrim", xs_ltrim, __FILE__);
        cv_set_call_checker(cv, ltrim_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::rtrim", xs_rtrim, __FILE__);
        cv_set_call_checker(cv, rtrim_call_checker, (SV*)cv);
    }

    /* Conditional ops with call checkers */
    {
        CV *cv = newXS("util::maybe", xs_maybe, __FILE__);
        cv_set_call_checker(cv, maybe_call_checker, (SV*)cv);
    }

    /* Numeric ops with call checkers */
    {
        CV *cv = newXS("util::sign", xs_sign, __FILE__);
        cv_set_call_checker(cv, sign_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::min2", xs_min2, __FILE__);
        cv_set_call_checker(cv, min2_call_checker, (SV*)cv);
    }
    {
        CV *cv = newXS("util::max2", xs_max2, __FILE__);
        cv_set_call_checker(cv, max2_call_checker, (SV*)cv);
    }

    Perl_xs_boot_epilog(aTHX_ ax);
}
