# Licensed to Elasticsearch B.V. under one or more contributor
# license agreements. See the NOTICE file distributed with
# this work for additional information regarding copyright
# ownership. Elasticsearch B.V. licenses this file to you under
# the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

require 'spec_helper'
require 'elastic-transport'
require_relative File.expand_path('../../utils/thor/endpoint_spec', __dir__)
require_relative File.expand_path('../../utils/thor/generator/files_helper', __dir__)

describe 'Perform request args' do
  Elasticsearch::API::FilesHelper.files.each do |filepath|
    spec = Elasticsearch::API::EndpointSpec.new(filepath)
    next if spec.module_namespace.flatten.first == '_internal' ||
            spec.visibility != 'public' ||
            # TODO: Once the test suite is migrated to elasticsearch-specification, these should be removed
            spec.module_namespace.flatten.first == 'rollup' ||
            [
              'scroll', 'clear_scroll', 'connector.last_sync', 'knn_search',
              'indices.remove_block'
            ].include?(spec.endpoint_name)

    # These are the path parts defined by the user in the method argument
    defined_path_parts = spec.path_params.inject({}) do |params, part|
      params.merge(part => 'testing')
    end

    # These are the required params, we must pass them to the method even when testing
    required_params = spec.required_parts.inject({}) do |params, part|
      params.merge(part.to_sym => 'testing')
    end

    required_params.merge!(body: {}) if ['inference.put', 'inference.update', 'inference.chat_completion_unified'].include? spec.endpoint_name

    let(:client_double) do
      Class.new { include Elasticsearch::API }.new.tap do |client|
        expect(client).to receive(:perform_request) do |_, _, _, _, _, request_params|
          # Check that the expected hash is passed to the perform_request method
          expect(request_params).to eq(expected_perform_request_params)
        end.and_return(response_double)
      end
    end

    let(:response_double) do
      double('response', status: 200, body: {}, headers: {})
    end

    let(:transport_double) do
      Transport ||= Struct.new('Transport', :options)
      Transport.new({ transport_options: { headers: {} } })
    end

    before do
      allow(client_double).to receive(:transport).and_return transport_double
      allow(Elasticsearch::API::Utils).to receive(:update_ndjson_headers!).and_return({})
    end

    context("'#{spec.endpoint_name}'") do
      # The expected hash passed to perform_request contains the endpoint name and any defined path parts
      let(:expected_perform_request_params) do
        if defined_path_parts.empty?
          { endpoint: spec.endpoint_name }
        else
          { endpoint: spec.endpoint_name, defined_params: defined_path_parts }
        end
      end

      if spec.path_parts.empty?
        it "passes the endpoint id to the request" do
          if spec.module_namespace.empty?
            client_double.send(spec.method_name, required_params)
          else
            client_double.send(spec.module_namespace[0]).send(spec.method_name, required_params)
          end
        end
      else
        it "passes params to the request with the endpoint id: #{spec.path_parts.keys}" do
          if spec.module_namespace.empty?
            client_double.send(spec.method_name, required_params.merge(defined_path_parts))
          else
            client_double.send(
              spec.module_namespace[0]).send(spec.method_name, required_params.merge(defined_path_parts)
            )
          end
        end
      end
    end
  end
end
