/* This file is part of ksirc
   Copyright (c) 2001 Malte Starostik <malte@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

// $Id: ksview.cpp,v 1.64.2.1 2002/12/13 22:40:30 hausmann Exp $

#include <qclipboard.h>
#include <qdatetime.h>
#include <qregexp.h>
#include <qdragobject.h>
#include <qvaluestack.h>

#include <kapplication.h>
#include <kdebug.h>
#include <klocale.h>
#include <krun.h>
#include <kpopupmenu.h>
#include <kstringhandler.h>
#include <kfiledialog.h>
#include <kio/job.h>

#include "ksopts.h"
#include "ksview.moc"

// Helper class to parse IRC colour/style codes and convert them to
// richtext. The parser maintains an internal stack of the styles
// applied because the IRC message could contain sequences as
// (bold)Hello (red)World(endbold)! (blue)blue text
// which needs to be converted to
// <b>Hello </b><font color="red"><b>World</b>! </font><font color="blue">blue text</font>
// to get correctly nested tags. (malte)
class KSParser
{
public:
    QString parse(const QString &);

private:
    QString pushTag(const QString &, const QString & = QString::null);
    QString popTag(const QString &);
    QString toggleTag(const QString &);
    QString popAll();
    QColor ircColor(int);

private:
    QValueStack<QString> m_tags;
    QMap<QString, QString> m_attributes;
};

QString KSParser::parse( const QString &message )
{
    QString res;
    m_tags.clear();
    m_attributes.clear();

    for (unsigned int i = 0; i < message.length();)
    {
        QChar ch = message[i++];

        if (ch.latin1() == 0x03 || ch == '~' && i < message.length())
        {
            QChar next = message[ i++ ];
            if (next.latin1() >= 0x30 && next.latin1() <= 0x39)
            {
                int fg = -1, len;
                int bg = -1;
                QRegExp colorRex("^[0-9]+");
                if (colorRex.search(message.mid(--i)) >= 0)
                {
                    len = colorRex.matchedLength();
                    fg = message.mid(i, len).toInt();
                    i += len;
                }
                if (message[i] == ',')
                {
                    if (colorRex.search(message.mid(++i)) >= 0)
                    {
                        len = colorRex.matchedLength();
                        bg = message.mid(i, len).toInt();
                        i += len;
                    }
                }
                QColor c = ircColor(fg);
                if ( c.isValid() )
                    res += pushTag( "font", QString( "color=\"%1\"" ).arg( c.name() ) );
                //else
                //        res += popTag( "font" );

                c = ircColor(bg);
                if ( c.isValid() )
                    res += pushTag( "font", QString( "bgcolor=\"%1\"" ).arg( c.name() ) );
            }
            else if (ch.latin1() == 0x03)
                res += popTag( "font" );
            else if (ch == '~')
            {
                switch (next)
                {
                case 'c':
                    res += popTag( "font" );
                    break;
                case 'C':
                    res += popAll();
                    break;
                case 'r': break;
                case 's': break;
                case 'b':
                    res += toggleTag( "b" );
                    break;
                case 'u':
                    res += toggleTag( "u" );
                    break;
                case 'n':
                    res += pushTag( "font", QString( "color=\"%1\"" ).arg( ksopts->nickForeground.name() ) );
                    break;
                case 'o':
                    res += pushTag( "font", QString( "color=\"%1\"" ).arg( ksopts->ownNickColor.name() ) );
                    break;
                case '~':
                    res += ch;
                }
            }
        }
        else
            res += ch;

    }

    res.append( popAll() );
    return res;
}

QString KSParser::pushTag(const QString &tag, const QString &attributes)
{
    QString res;
    m_tags.push(tag);
    if (!m_attributes.contains(tag))
        m_attributes.insert(tag, attributes);
    else if (!attributes.isEmpty())
        m_attributes.replace(tag, attributes);
    res.append("<" + tag);
    if (!m_attributes[tag].isEmpty())
        res.append(" " + m_attributes[tag]);
    return res + ">";
}

QString KSParser::popTag(const QString &tag)
{
    if (!m_tags.contains(tag))
        return QString::null;

    QString res;
    QValueStack<QString> savedTags;
    while (m_tags.top() != tag)
    {
        savedTags.push(m_tags.pop());
        res.append("</" + savedTags.top() + ">");
    }
    res.append("</" + m_tags.pop() + ">");
    m_attributes.remove(tag);
    while (!savedTags.isEmpty())
        res.append(pushTag(savedTags.pop()));
    return res;
}

QString KSParser::toggleTag(const QString &tag)
{
    return m_attributes.contains(tag) ? popTag(tag) : pushTag(tag);
}

QString KSParser::popAll()
{
    QString res;
    while (!m_tags.isEmpty())
        res.append("</" + m_tags.pop() + ">");
    m_attributes.clear();
    return res;
}

QColor KSParser::ircColor(int code)
{
    if (code >= 0 && code < 16)
        return ksopts->ircColors[code];
    return QColor();
}

KSircView::KSircView(QWidget *parent, const char *name)
    : KSirc::TextView(parent, name)
{
    m_acceptFiles = false;
    viewport()->setAcceptDrops(true);
    clear();
    connect( this, SIGNAL( linkClicked( const QMouseEvent *, const QString & ) ),
             this, SLOT( anchorClicked( const QMouseEvent *, const QString & ) ) );

	QPixmap background = ksopts->backgroundPixmap();
	if ( !background.isNull() )
		viewport()->setBackgroundPixmap( background );
}

KSircView::~KSircView()
{
}

void KSircView::clear()
{
    m_lines = 0;
    m_timeStamps.clear();
	KSirc::TextView::clear();
}

QString KSircView::makeTimeStamp()
{
    QTime now = QTime::currentTime();
    QString timeStamp = QString::fromLatin1( "[%1:%2:%3] " )
        .arg( QString::number( now.hour() ).rightJustify( 2, '0' ) )
        .arg( QString::number( now.minute() ).rightJustify( 2, '0' ) )
        .arg( QString::number( now.second() ).rightJustify( 2, '0' ) );
    return timeStamp; 
}

void KSircView::saveURL( const QString &url )
{
	KURL kurl( url );

	KFileDialog *dlg = new KFileDialog( QString::null, QString::null /*filter*/, this /*parent*/, "filedialog" /*name*/, true /*modal*/ );

	dlg->setKeepLocation( true );

	dlg->setCaption( i18n( "Save As" ) );

	if ( !kurl.fileName().isEmpty() )
		dlg->setSelection( kurl.fileName() );

	if ( dlg->exec() ) {
		KURL destURL( dlg->selectedURL() );
		if ( !destURL.isMalformed() ) {
			KIO::Job *job = KIO::copy( kurl, destURL );
			job->setAutoErrorHandlingEnabled( true );
		}
	}

	delete dlg;
}

QString KSircView::addLine(const QString &pixmap, const QColor &color, const QString &_text)
{
    QString richText( "<font color=\"%1\">" );
    richText = richText.arg( color.name() );

    if ( !pixmap.isEmpty() )
        richText.prepend( QString::fromLatin1( "<img src=\"%1\"></img>" ).arg( pixmap ) );

    QString timeStamp = QString::fromLatin1( "<font color=\"%1\">%2</font>" )
        .arg( ksopts->textColor.name() )
        .arg( makeTimeStamp() );
    m_timeStamps.append(timeStamp);
    if ( ksopts->timeStamp )
        richText.prepend( timeStamp );

    QRegExp ampRex("&");
    QRegExp ltRex("<");
    QRegExp gtRex(">");
    QString text = _text;
    text.replace(ampRex, "&amp;");
    text.replace(ltRex, "&lt;");
    text.replace(gtRex, "&gt;");

    // ### a bit of a hack: turn '&lt;nick&gt; message' into
    // <span>&lt;nick&gt;<span> message' . span itself isn't supported but it
    // enforces the creation of separate item objects and hence separate
    // drawing of '<nick>' and 'message' , which is needed for BiDi users,
    // according to UV Kochavi <uv1st@yahoo.com> , to prevent output like
    // '<nick message<' , which is supposedly a bug in Qt's reordering.  The
    // same is done for [nick] and >nick< to catch queries.
    QRegExp bidiRe( "^(&lt;\\S+&gt;)(.+)$" );
    text.replace( bidiRe, QString::fromLatin1( "<span>\\1</span>\\2" ) );
    QRegExp bidiRe2( "^(\\[\\S+\\])(.+)$" );
    text.replace( bidiRe2, QString::fromLatin1( "<span>\\1</span>\\2" ) );
    QRegExp bidiRe3( "^(&gt;\\S+&lt;)(.+)$" );
    text.replace( bidiRe3, QString::fromLatin1( "<span>\\1</span>\\2" ) );

    KSParser parser;
    richText += parser.parse( text );

    richText += "</font>";

	richText = KStringHandler::tagURLs( richText );
	KSirc::TextParagIterator parag = appendParag( richText );

    m_lines++;
    if ( ksopts->windowLength && m_lines > ksopts->windowLength )
    {
        while ( m_lines > ksopts->windowLength )
        {
            removeParag( firstParag() );
            m_timeStamps.remove( m_timeStamps.begin() );
            m_lines--;
        }
    }

    QString logText = parag.plainText();
    // append timestamp if it's not already there
    if ( !ksopts->timeStamp )
        logText.prepend( makeTimeStamp() );

    return logText + '\n';
}

void KSircView::enableTimeStamps(bool enable)
{
    setUpdatesEnabled( false );
	KSirc::TextParagIterator paragIt = firstParag();
    QStringList::ConstIterator timeStampIt = m_timeStamps.begin();
    for (; !paragIt.atEnd(); ++paragIt, ++timeStampIt )
    {
        QString text = paragIt.richText();
        if ( enable )
            text.prepend( *timeStampIt );
        else
            text.remove( 0, (*timeStampIt).length() );
        paragIt.setRichText( text );
    }
    setUpdatesEnabled( true );
    updateContents();
}

void KSircView::anchorClicked(const QMouseEvent *ev, const QString &url)
{
    if ( (ev->button() & LeftButton) && (ev->state() & ShiftButton ) )
		saveURL( url );
	else if ( (ev->button() & LeftButton) || (ev->button() & MidButton) )
    {
        openBrowser( url );
    }
    else if ( ev->button() & RightButton )
    {
        static const int openURLID = 0;
        static const int copyLinkLocationID = 1;

        // Adding a nice contextmenu
        KPopupMenu* menu = new KPopupMenu( this );
        menu->insertTitle( i18n( "URL" ) );
        menu->insertItem( i18n("Open URL"), openURLID );
        menu->insertItem( i18n("Copy Link Location"), copyLinkLocationID );
        switch( menu->exec( ( ev->globalPos() ) ) )
        {
        case openURLID :
            openBrowser( url );
            break;
        case copyLinkLocationID :
            copyLinkToClipboard( url );
            break;
        default:
            break;
        }
        delete menu;
    }
}

void KSircView::openBrowser(const QString &url )
{
    (void) new KRun( url.startsWith("www") ? QString::fromLatin1("http://") + url : url);
}

void KSircView::copyLinkToClipboard( const QString &url )
{
    QClipboard *clip = QApplication::clipboard();
    bool oldMode = clip->selectionModeEnabled();
    clip->setSelectionMode( false );
    clip->setText( url );
    clip->setSelectionMode( oldMode );
}

QColor KSircView::ircColor(int code)
{
    if (code >= 0 && code < 16)
        return ksopts->ircColors[code];
    return QColor();
}

void KSircView::contentsDragEnterEvent(QDragEnterEvent* event)
{
    event->accept((QTextDrag::canDecode(event) ||
                   (m_acceptFiles && QUriDrag::canDecode(event))) &&
                  (!event->source() || event->source() != viewport()));
}

void KSircView::contentsDragMoveEvent(QDragMoveEvent* event)
{
    event->accept(!event->source() || event->source() != viewport());
}

void KSircView::contentsDropEvent(QDropEvent* event)
{
    QStringList urls;
    QString text;

    if (m_acceptFiles && QUriDrag::decodeLocalFiles(event, urls))
        emit urlsDropped(urls);
    else if (QTextDrag::decode(event, text))
        emit textDropped(text);
}

// vim: ts=4 sw=4 noet
