/*
 * @(#)LineCountMeasure.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.module;


import java.util.HashSet;

import net.sourceforge.groboutils.codecoverage.v2.IAnalysisMetaData;
import net.sourceforge.groboutils.codecoverage.v2.IMethodCode;

import org.apache.bcel.classfile.LineNumber;
import org.apache.bcel.classfile.LineNumberTable;


/**
 * Processes methods for line-count coverage analysis.  Currently, this
 * does not support localization.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     December 17, 2002
 * @see       IAnalysisMetaData
 */
public class LineCountMeasure extends AbstractMeasure
{
    private static final org.apache.log4j.Logger LOG =
        org.apache.log4j.Logger.getLogger( LineCountMeasure.class );
    
    /**
     * Returns the human-readable name of the measure.
     */
    public String getMeasureName()
    {
        return "LineCount";
    }
    
    /**
     * Returns the unit name for this particular coverage measure.
     */
    public String getMeasureUnit()
    {
        return "lines";
    }
    
    
    /**
     * Returns the text format used in meta-data formatted text.  This should
     * be the mime encoding type, such as "text/plain" or "text/html".
     */
    public String getMimeEncoding()
    {
        return "text/plain";
    }
    
    
    /**
     * Perform the analysis on the method.
     */
    public void analyze( IMethodCode method )
    {
        //Method m = method.getOriginalMethod();
        LineNumberTable lnt = method.getLineNumberTable();
        if (lnt == null)
        {
            // nothing to do
            LOG.info( "Method "+method+" has no line numbers." );
            return;
        }
        
        // we must ensure that the same line number is not counted
        // twice.
        HashSet linesFound = new HashSet();
        
        LineNumber[] lines = lnt.getLineNumberTable();
        int maxInstructionCount = method.getInstructionCount();
        int instructionPos[] = new int[ maxInstructionCount ];
        
        // find the positions of the instructions in the bytecode
        int methodSize = 0;
        for (int i = 0; i < maxInstructionCount; ++i)
        {
            instructionPos[i] = methodSize;
            methodSize += method.getInstructionAt( i ).getLength();
        }
        
        for (int i = 0; i < lines.length; ++i)
        {
            int bytecodeOffset = lines[i].getStartPC();
            for (int j = 1; j < maxInstructionCount; ++j)
            {
                if (bytecodeOffset < instructionPos[j])
                {
                    Integer lineNo = new Integer( lines[i].getLineNumber() );
                    if (!linesFound.contains( lineNo ) )
                    {
                        // the bytecode start was one up from the current
                        // instruction position.
                        IAnalysisMetaData amd = createAnalysisMetaData(
                            lines[i] );
                        
                        // Bug 906207
                        markInstruction( method, j-1, amd, false );
                        
                        // ensure we don't find this line number again
                        linesFound.add( lineNo );
                    }
                    
                    break;
                }
            }
        }
    }
    
    
    private IAnalysisMetaData createAnalysisMetaData( LineNumber ln )
    {
        int lineNo = ln.getLineNumber();
        return new DefaultAnalysisMetaData(
            "Line "+lineNo,
            "Didn't cover line "+lineNo,
            (byte)0 );
    }
}

