/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 * Copyright (C) 2007-2008  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#ifndef _SML_TRANSPORT_INTERNALS_H_
#define _SML_TRANSPORT_INTERNALS_H_

#include "sml_queue_internals.h"

typedef SmlBool (* SmlTransportSetConfigOptionFn) (SmlTransport *tsp, const char *name, const char *value, SmlError **error);
typedef SmlBool (* SmlTransportSetConnectionTypeFn) (SmlTransport *tsp, SmlTransportConnectionType type, SmlError **error);
typedef SmlBool (* SmlTransportInitializeFn) (SmlTransport *tsp, SmlError **error);

/* This function is used for transport clients
 * to support redirects from the server.
 * Example: Funambol works with such redirects
 *          because of Tomcat's session management.
 */
typedef SmlBool (* SmlTransportSetResponseUriFn) (SmlTransport *tsp, const char *uri, SmlError **error);

/* This function is used to add some kind of session id
 * to the URL of the server. This is necessary to
 * avoid session hijacking.
 */
typedef char * (* SmlTransportGetResponseUriFn) (SmlLink *link, SmlSession *session, SmlError **error);

typedef SmlBool (* SmlTransportFinalizeFn) (void *data, SmlError **error);

typedef void (* SmlTransportConnectFn) (void *data);
typedef void (* SmlTransportDisconnectFn) (void *data, void *link_data);
typedef void (* SmlTransportSendFn) (void *userdata, void *link_data, SmlTransportData *data, SmlError *error);

typedef struct SmlTransportFunctions {
	SmlTransportSetConfigOptionFn set_config_option;
	SmlTransportSetConnectionTypeFn set_connection_type;
	SmlTransportInitializeFn initialize;
	SmlTransportSetResponseUriFn set_response_uri;
	SmlTransportGetResponseUriFn get_response_uri;
	SmlTransportFinalizeFn finalize;
	SmlTransportConnectFn connect;
	SmlTransportDisconnectFn disconnect;
	SmlTransportSendFn send;
} SmlTransportFunctions;

typedef enum SmlTransportState {
	SML_TRANSPORT_UNINITIALIZED,
	SML_TRANSPORT_INITIALIZED,
	SML_TRANSPORT_CONNECTED,
	SML_TRANSPORT_ERROR
} SmlTransportState;

struct SmlTransport {
	GMainContext *context;
	SmlThread *thread;
	
	SmlTransportState state;
	
	SmlTransportType type;
	SmlTransportFunctions functions;
	void *transport_data;
	
	SmlQueue *command_queue;
	SmlTransportEventCb event_callback;
	void *event_callback_userdata;
	gint event_callback_ref_count;
	SmlError *cached_error;
	
	SmlBool connected;
	GHashTable *links;
	GMutex *links_mutex;
	unsigned int connections;
	GMutex *connections_mutex;
};

struct SmlLink {
	SmlTransport *tsp;
	void *link_data;
	gint32 refCount;
};

typedef enum SmlTransportCommandType {
	SML_TRANSPORT_CMD_SEND,
	SML_TRANSPORT_CMD_CONNECT,
	SML_TRANSPORT_CMD_DISCONNECT
} SmlTransportCommandType;

typedef struct SmlTransportCommand {
	SmlTransportCommandType type;
	SmlTransportData *data;
	const void *config;
	SmlLink *link;
	SmlError *error;
} SmlTransportCommand;

struct SmlTransportData {
	char *data;
	unsigned long size;
	SmlMimeType type;
	SmlBool ownsData;
	gint32 refCount;
	/** This field specifies if the data transported
	 * here needs an reply from the other side. If it does not
	 * need one, the transports might take special precautions */
	SmlBool needsAnswer;
	/** This is only needed to switch the mime type when using the OBEX Transporting
	 * and sending a SyncML 1.2 SAN package. The transport send() function for OBEX transport
	 * make use of a PUT/GET command sequence. To avoid that GET command request with mimetype
	 * SAN, we have to store the regular mimetype to continue the sync in this struct.
	 * So the SyncML 1.2 notification looks like this:
	 * PUT SML_MIMETYPE_SAN
	 * -- wait for response --
	 * GET SML_MIMETYPE_XML / SML_MIMETYPE_WBXML (not SML_MIMETYPE_SAN!)
	 * -- wait for resposne --
	 */
	SmlMimeType type_get;
};

void smlTransportWorkerHandler(void *message, void *userdata);
SmlBool smlTransportReceiveEvent(SmlTransport *tsp, SmlLink *link, SmlTransportEventType type, SmlTransportData *data, SmlError *error);

SmlBool smlTransportSetResponseURI(SmlTransport *tsp, const char *uri, SmlError **error);
char *smlTransportGetResponseURI(SmlLink *link, SmlSession *session, SmlError **error);

#endif //_SML_TRANSPORT_INTERNALS_H_
